/**
 * ChatTOC - Content Script 主文件（纯 JavaScript）
 * 侧边栏 UI 与解析逻辑集中在此文件中，使用 Shadow DOM 注入 UI。
 */

// Content script loaded.

// ==================== 工具函数：文本处理 ====================
function cleanText(text) {
  if (!text) return ''

  let cleaned = text
  cleaned = cleaned.replace(/^#{1,6}\s+/gm, '')
  cleaned = cleaned.replace(/^[-*+]\s+/gm, '')
  cleaned = cleaned.replace(/```[\s\S]*?```/g, '')
  cleaned = cleaned.replace(/`[^`]+`/g, '')
  cleaned = cleaned.replace(/\n{3,}/g, '\n\n')
  cleaned = cleaned.trim()

  return cleaned
}

function truncateText(text, maxLength) {
  if (!text) return ''
  if (text.length <= maxLength) return text

  const truncated = text.slice(0, maxLength)
  const lastSpace = truncated.lastIndexOf(' ')
  if (lastSpace > maxLength * 0.7) {
    return truncated.slice(0, lastSpace) + '...'
  }
  return truncated + '...'
}

function escapeHtml(text) {
  const div = document.createElement('div')
  div.textContent = text
  return div.innerHTML
}

function highlightText(text, query) {
  if (!query) return escapeHtml(text)

  const lowerText = text.toLowerCase()
  const lowerQuery = query.toLowerCase()
  const queryLength = query.length
  let result = ''
  let startIndex = 0
  let matchIndex = lowerText.indexOf(lowerQuery)

  while (matchIndex !== -1) {
    result += escapeHtml(text.slice(startIndex, matchIndex))
    result += `<mark>${escapeHtml(text.slice(matchIndex, matchIndex + queryLength))}</mark>`
    startIndex = matchIndex + queryLength
    matchIndex = lowerText.indexOf(lowerQuery, startIndex)
  }

  result += escapeHtml(text.slice(startIndex))
  return result
}

function debounce(fn, delay) {
  let timer = null
  return (...args) => {
    if (timer) {
      clearTimeout(timer)
    }
    timer = window.setTimeout(() => {
      fn(...args)
      timer = null
    }, delay)
  }
}

function isEditableElement(target) {
  if (!target) return false
  const tagName = target.tagName?.toLowerCase()
  return tagName === 'input' || tagName === 'textarea' || target.isContentEditable
}

function getFaviconUrl() {
  const selectors = [
    'link[rel="icon"]',
    'link[rel="shortcut icon"]',
    'link[rel="apple-touch-icon"]',
    'link[rel="apple-touch-icon-precomposed"]'
  ]

  for (const selector of selectors) {
    const link = document.querySelector(selector)
    const href = link?.getAttribute('href')
    if (!href) continue
    try {
      return new URL(href, window.location.origin).href
    } catch (error) {
      return href
    }
  }

  return ''
}

function normalizeBookmarked(value) {
  if (value === true || value === false) return value
  if (value === 'true' || value === 1 || value === '1') return true
  return false
}

// ==================== 默认设置 ====================
const sharedConfig = globalThis.ChatTOCConfig || {}
const defaultSettings = sharedConfig.defaultSettings || {}

const PREVIEW_MAX_LENGTH = 140
const VIRTUAL_ROW_HEIGHT = 82
const VIRTUAL_OVERSCAN = 6
const SIDEBAR_MIN_WIDTH = 240
const SIDEBAR_MAX_WIDTH = 520
const KEY_STORAGE_BOOKMARKS = 'chattoc.bookmarks'
const KEY_STORAGE_SIDEBAR_STATE = 'chattoc.sidebarState'

const SITE_LABELS = sharedConfig.siteLabels || {
  'chat.openai.com': { name: 'ChatGPT', icon: '🤖' },
  'chatgpt.com': { name: 'ChatGPT', icon: '🤖' },
  'gemini.google.com': { name: 'Gemini', icon: '🤖' },
  'claude.ai': { name: 'Claude', icon: '🤖' },
  'chat.deepseek.com': { name: 'DeepSeek', icon: '🤖' },
  'www.doubao.com': { name: '豆包', icon: '🤖' },
  'grok.com': { name: 'Grok', icon: '🤖' },
  'grok.x.ai': { name: 'Grok', icon: '🤖' },
  'x.ai': { name: 'Grok', icon: '🤖' }
}

const I18N = {
  zh: {
    siteStatus: '站点状态',
    collapse: '收起侧边栏',
    close: '关闭侧边栏',
    open: '打开侧边栏',
    expand: '展开侧边栏',
    refresh: '刷新消息',
    export: '导出',
    exportJson: '导出 JSON',
    exportMarkdown: '导出 Markdown',
    exportMenu: '导出菜单',
    settings: '设置',
    clear: '清除',
    bookmark: '收藏',
    searchPlaceholder: '搜索消息内容 / 关键词',
    filterAll: '全部',
    filterUser: '用户',
    filterAssistant: 'AI',
    filterBookmark: '★ 收藏',
    userLabel: '用户',
    assistantLabel: 'Assistant',
    loading: '⏳ 正在解析对话…',
    unsupported: '⚠ 当前页面暂不支持',
    degraded: '⚠ 检测到站点更新，部分功能可能受限',
    empty: '暂无消息',
    emptyFiltered: '没有匹配的消息'
  },
  en: {
    siteStatus: 'Site status',
    collapse: 'Collapse sidebar',
    close: 'Close sidebar',
    open: 'Open sidebar',
    expand: 'Expand sidebar',
    refresh: 'Refresh messages',
    export: 'Export',
    exportJson: 'Export JSON',
    exportMarkdown: 'Export Markdown',
    exportMenu: 'Export menu',
    settings: 'Settings',
    clear: 'Clear',
    bookmark: 'Bookmark',
    searchPlaceholder: 'Search messages / keywords',
    filterAll: 'All',
    filterUser: 'User',
    filterAssistant: 'AI',
    filterBookmark: '★ Favorites',
    userLabel: 'User',
    assistantLabel: 'Assistant',
    loading: '⏳ Parsing conversation…',
    unsupported: '⚠ Page not supported',
    degraded: '⚠ Site update detected, some features limited',
    empty: 'No messages yet',
    emptyFiltered: 'No matching messages'
  }
}

function resolveLanguage(lang) {
  if (lang === 'auto') {
    const locale = navigator.language || 'zh'
    return locale.toLowerCase().startsWith('zh') ? 'zh' : 'en'
  }
  return lang || 'zh'
}

// ==================== 工具类：存储管理 ====================
function normalizeSettings(source) {
  return {
    ...defaultSettings,
    ...source,
    general: { ...defaultSettings.general, ...(source.general || {}) },
    labels: { ...defaultSettings.labels, ...(source.labels || {}) },
    appearance: { ...defaultSettings.appearance, ...(source.appearance || {}) },
    rules: { ...defaultSettings.rules, ...(source.rules || {}) },
    platforms: { ...defaultSettings.platforms, ...(source.platforms || {}) },
    customSites: source.customSites !== undefined
      ? source.customSites
      : defaultSettings.customSites
  }
}

class StorageManager {
  static async getSettings() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['settings'], (result) => {
        resolve(normalizeSettings(result.settings || {}))
      })
    })
  }

  static async saveSettings(settings) {
    return new Promise((resolve) => {
      chrome.storage.sync.set({ settings }, () => {
        resolve()
      })
    })
  }

  static async updateSettings(updates) {
    const current = await this.getSettings()
    const updated = {
      general: { ...current.general, ...(updates.general || {}) },
      labels: { ...current.labels, ...(updates.labels || {}) },
      appearance: { ...current.appearance, ...(updates.appearance || {}) },
      rules: { ...current.rules, ...(updates.rules || {}) },
      platforms: { ...current.platforms, ...(updates.platforms || {}) },
      customSites: updates.customSites !== undefined
        ? updates.customSites
        : current.customSites
    }
    await this.saveSettings(updated)
  }

  static onSettingsChange(callback) {
    const listener = (changes) => {
      if (changes.settings) {
        callback(changes.settings.newValue)
      }
    }
    chrome.storage.onChanged.addListener(listener)
    return () => chrome.storage.onChanged.removeListener(listener)
  }
}

class LocalStateManager {
  static async get(key) {
    return new Promise((resolve) => {
      chrome.storage.local.get([key], (result) => {
        resolve(result[key] || {})
      })
    })
  }

  static async set(key, value) {
    return new Promise((resolve) => {
      chrome.storage.local.set({ [key]: value }, () => resolve())
    })
  }

  static async update(key, updater) {
    const current = await this.get(key)
    const next = updater({ ...current })
    await this.set(key, next)
    return next
  }
}

// ==================== 工具类：观察器管理 ====================
class ObserverManager {
  constructor(callback, debounceDelay = 300) {
    this.observer = null
    this.callback = callback
    this.debounceTimer = null
    this.debounceDelay = debounceDelay
  }

  observe(target, options = {}) {
    if (this.observer) {
      this.disconnect()
    }

    const defaultOptions = {
      childList: true,
      subtree: true,
      characterData: true,
      ...options
    }

    this.observer = new MutationObserver(() => {
      this.debouncedCallback()
    })

    this.observer.observe(target, defaultOptions)
  }

  disconnect() {
    if (this.observer) {
      this.observer.disconnect()
      this.observer = null
    }
    if (this.debounceTimer !== null) {
      clearTimeout(this.debounceTimer)
      this.debounceTimer = null
    }
  }

  debouncedCallback() {
    if (this.debounceTimer !== null) {
      clearTimeout(this.debounceTimer)
    }

    this.debounceTimer = window.setTimeout(() => {
      this.callback()
      this.debounceTimer = null
    }, this.debounceDelay)
  }
}

// ==================== 站点配置 ====================
const sitesConfig = {
  'chat.openai.com': {
    site_name: 'ChatGPT',
    domain: 'chat.openai.com',
    selectors: {
      container: 'main',
      item: "[data-testid*='conversation-turn']",
      userMessage: "[data-message-author-role='user']",
      text: "[data-message-author-role='user'] .markdown",
      exclude: "[data-testid*='error']"
    }
  },
  'chatgpt.com': {
    site_name: 'ChatGPT',
    domain: 'chatgpt.com',
    selectors: {
      container: 'main',
      item: "[data-testid*='conversation-turn']",
      userMessage: "[data-message-author-role='user']",
      text: "[data-message-author-role='user'] .markdown",
      exclude: "[data-testid*='error']"
    }
  },
  'gemini.google.com': {
    site_name: 'Gemini',
    domain: 'gemini.google.com',
    selectors: {
      container: 'chat-window-content',
      item: '.conversation-container',
      userMessage: 'user-query',
      text: 'user-query .query-text, model-response .markdown, model-response .markdown-main-panel',
      exclude: '.error'
    }
  },
  'claude.ai': {
    site_name: 'Claude',
    domain: 'claude.ai',
    selectors: {
      container: 'main, [role="main"], #root, div.overflow-y-scroll.overflow-x-hidden.pt-6.flex-1',
      item: '[data-testid="user-message"], .standard-markdown',
      userMessage: '[data-testid="user-message"]',
      text: '[data-testid="user-message"], .standard-markdown',
      exclude: '[data-testid="message-warning"], [class*="Error"]'
    }
  },
  'chat.deepseek.com': {
    site_name: 'DeepSeek',
    domain: 'chat.deepseek.com',
    selectors: {
      container: 'main, [role=\'main\'], #root, .app-container',
      item: '.ds-message',
      userMessage: '.ds-message',
      text: '.ds-message',
      exclude: "[class*='error'], [class*='Error']"
    }
  },
  'grok.com': {
    site_name: 'Grok',
    domain: 'grok.com',
    selectors: {
      container: 'main, [role="main"], #root, body',
      item: 'div[id^="response-"]',
      userMessage: '.items-end',
      text: '.response-content-markdown',
      exclude: ''
    }
  },
  'grok.x.ai': {
    site_name: 'Grok',
    domain: 'grok.x.ai',
    selectors: {
      container: 'main, [role="main"], #root, body',
      item: 'div[id^="response-"]',
      userMessage: '.items-end',
      text: '.response-content-markdown',
      exclude: ''
    }
  },
  'x.ai': {
    site_name: 'Grok',
    domain: 'x.ai',
    selectors: {
      container: 'main, [role="main"], #root, body',
      item: 'div[id^="response-"]',
      userMessage: '.items-end',
      text: '.response-content-markdown',
      exclude: ''
    }
  },
  'www.doubao.com': {
    site_name: '豆包',
    domain: 'www.doubao.com',
    selectors: {
      container: '.inter-H_fm37, main, [role=\'main\'], #root',
      item: "[data-testid='message_content']",
      userMessage: "[data-testid='send_message']",
      text: "[data-testid='message_text_content']",
      exclude: "[class*='error'], [class*='Error']"
    }
  }
}

const CHATGPT_FALLBACK_SELECTORS = {
  container: [
    'main',
    '[role="main"]',
    '#__next > div > div',
    'div[class*="conversation"]',
    'div[class*="chat"]'
  ],
  item: [
    '[data-testid*="conversation-turn"]',
    '[data-testid*="turn"]',
    'div[class*="conversation-turn"]',
    'div[class*="message"]',
    'div[role="article"]'
  ],
  text: [
    '[data-message-author-role="user"] .markdown',
    '[data-message-author-role="user"]',
    '.markdown',
    '[class*="markdown"]',
    'div[class*="text"]'
  ]
}

const CLAUDE_FALLBACK_SELECTORS = {
  container: [
    'main',
    '[role="main"]',
    '#root',
    'div[class*="Chat"]',
    'div[class*="Conversation"]'
  ],
  item: [
    '[data-testid="user-message"]',
    '.standard-markdown',
    '[data-testid="chat-message"]',
    '[data-testid="chat-message-content"]',
    '[data-message-author-role]',
    '[class*="Message"]',
    '[class*="ChatMessage"]'
  ],
  text: [
    '[data-testid="user-message"]',
    '.standard-markdown',
    '.prose',
    '[data-testid="chat-message-content"]',
    '[class*="Message"] [class*="Text"]',
    '[class*="Message"]',
    '[class*="content"]'
  ]
}

function isChatGPTDomain(domain) {
  return domain.includes('chatgpt') || domain.includes('openai')
}

function isGrokDomain(domain) {
  return domain === 'grok.com' || domain === 'grok.x.ai' || domain === 'x.ai'
}

// ==================== 工具类：解析器适配器 ====================
class ParserAdapter {
  constructor(domain) {
    this.config = null
    this.domain = domain
    this.loadConfig()
  }

  loadConfig() {
    const presetConfig = sitesConfig[this.domain]
    if (presetConfig) {
      this.config = presetConfig
    }
  }

  setCustomConfig(config) {
    this.config = config
  }

  isSupported() {
    return this.config !== null
  }

  findWithFallback(primarySelector, fallbackSelectors, queryFn) {
    const primary = queryFn(primarySelector)
    if (primary) return primary

    if (isChatGPTDomain(this.domain)) {
      for (const selector of fallbackSelectors) {
        const element = queryFn(selector)
        if (element) {
          // Fallback selector matched.
          return element
        }
      }
    }

    return null
  }

  getContainer() {
    if (!this.config) return null

    try {
      const container = this.findWithFallback(
        this.config.selectors.container,
        CHATGPT_FALLBACK_SELECTORS.container,
        (selector) => document.querySelector(selector)
      )

      if (container) {
        // Container matched.
      } else {
        console.warn('[ChatTOC] 未找到容器，选择器:', this.config.selectors.container)
      }

      return container
    } catch (e) {
      console.error('[ChatTOC] Invalid container selector:', e)
      return null
    }
  }

  getMessageElements() {
    if (!this.config) return []

    try {
      let container = this.getContainer()
      if (!container) {
        if (this.domain === 'claude.ai') {
          container = document.body
        } else if (isGrokDomain(this.domain)) {
          container = document.body
        } else {
          console.warn('[ChatTOC] 容器未找到，无法获取消息元素')
          return []
        }
      }

      if (this.domain === 'gemini.google.com') {
        const conversationContainers = Array.from(
          container.querySelectorAll('.conversation-container')
        )
        const items = []

        for (const convContainer of conversationContainers) {
          const userQuery = convContainer.querySelector('user-query')
          if (userQuery) {
            items.push(userQuery)
          }

          const modelResponse = convContainer.querySelector('model-response')
          if (modelResponse) {
            items.push(modelResponse)
          }
        }

        // Gemini message elements found.
        return items
      }

      let items = []
      const primaryItems = Array.from(
        container.querySelectorAll(this.config.selectors.item)
      )

      if (primaryItems.length > 0) {
        items = primaryItems
        // Message elements found.
        return items
      }

      if (this.domain === 'claude.ai') {
        for (const selector of CLAUDE_FALLBACK_SELECTORS.item) {
          const fallbackItems = Array.from(container.querySelectorAll(selector))
          if (fallbackItems.length > 0) {
            items = fallbackItems
            break
          }
        }
      }

      if (isChatGPTDomain(this.domain)) {
        for (const selector of CHATGPT_FALLBACK_SELECTORS.item) {
          const fallbackItems = Array.from(container.querySelectorAll(selector))
          if (fallbackItems.length > 0) {
            // Fallback message selector matched.
            items = fallbackItems
            break
          }
        }
      }

      return items
    } catch (e) {
      console.error('[ChatTOC] Error getting message elements:', e)
      return []
    }
  }

  isUserMessage(element) {
    if (!this.config) return false

    try {
      const roleAttr = element.getAttribute('data-message-author-role') ||
        element.getAttribute('data-author') ||
        element.getAttribute('data-role')
      if (roleAttr) {
        return roleAttr === 'user'
      }

      const roleNode = element.closest('[data-message-author-role],[data-author],[data-role]')
      if (roleNode) {
        const closestRole = roleNode.getAttribute('data-message-author-role') ||
          roleNode.getAttribute('data-author') ||
          roleNode.getAttribute('data-role')
        if (closestRole) {
          return closestRole === 'user'
        }
      }

      if (this.domain === 'gemini.google.com') {
        return element.tagName.toLowerCase() === 'user-query'
      }

      if (isGrokDomain(this.domain)) {
        const userContainer = element.classList.contains('items-end')
          ? element
          : element.closest('.items-end')
        if (userContainer) {
          return true
        }

        const assistantContainer = element.classList.contains('items-start')
          ? element
          : element.closest('.items-start')
        if (assistantContainer) {
          return false
        }
      }

      if (this.domain === 'www.doubao.com') {
        const dataTestId = element.getAttribute('data-testid') || ''
        if (dataTestId === 'send_message') {
          return true
        }
        if (dataTestId === 'receive_message') {
          return false
        }
        return Boolean(element.closest('[data-testid="send_message"]'))
      }

      if (this.domain === 'chat.deepseek.com') {
        const className = (element.className || '').toString().toLowerCase()
        if (/(assistant|bot|ai|model)/.test(className)) {
          return false
        }
        if (/(user|human|prompt)/.test(className)) {
          return true
        }
        if (element.closest('._4f9bf79, ._43c05b5') || element.querySelector('.ds-markdown') || element.closest('.ds-markdown')) {
          return false
        }
        if (element.closest('._9663006') || element.querySelector('.fbb737a4') || element.closest('.fbb737a4')) {
          return true
        }
        const roleAttr = element.getAttribute('data-role') ||
          element.getAttribute('data-author-role') ||
          element.getAttribute('data-message-author-role')
        if (roleAttr) {
          return roleAttr === 'user'
        }
        const roleNode = element.closest('[data-role],[data-author-role],[data-message-author-role]')
        if (roleNode) {
          const closestRole = roleNode.getAttribute('data-role') ||
            roleNode.getAttribute('data-author-role') ||
            roleNode.getAttribute('data-message-author-role')
          if (closestRole) {
            return closestRole === 'user'
          }
        }
      }

      if (this.domain === 'claude.ai') {
        return element.matches('[data-testid="user-message"]') ||
          Boolean(element.closest('[data-testid="user-message"]'))
      }

      return element.matches(this.config.selectors.userMessage) ||
        !!element.querySelector(this.config.selectors.userMessage)
    } catch (e) {
      console.error('[ChatTOC] Error checking user message:', e)
      return false
    }
  }

  extractText(element) {
    if (!this.config) return ''

    try {
      if (this.domain === 'gemini.google.com') {
        if (element.tagName.toLowerCase() === 'user-query') {
          const queryText = element.querySelector('.query-text')
          return queryText?.textContent || element.textContent || ''
        }

        if (element.tagName.toLowerCase() === 'model-response') {
          const markdown = element.querySelector('.markdown-main-panel') ||
            element.querySelector('.markdown') ||
            element.querySelector('.model-response-text')
          if (markdown) {
            return markdown.textContent || ''
          }
          return element.textContent || ''
        }
      }

      if (isGrokDomain(this.domain)) {
        const textElement = element.querySelector('.response-content-markdown') ||
          element.querySelector('.message-bubble')
        if (textElement) {
          return textElement.textContent || ''
        }
        return element.textContent || ''
      }

      if (this.domain === 'www.doubao.com') {
        let textElement = element.querySelector('[data-testid="message_text_content"]')

        if (!textElement && element.matches('[data-testid="message_content"]')) {
          textElement = element.querySelector('[data-testid="message_text_content"]')
        }

        if (!textElement && (element.matches('[data-testid="send_message"]') || element.matches('[data-testid="receive_message"]'))) {
          textElement = element.querySelector('[data-testid="message_text_content"]')
        }

        if (!textElement) {
          textElement = element.querySelector('[data-testid="message_text_content"]')
        }

        if (textElement) {
          const text = textElement.textContent || textElement.innerText || ''
          if (text.trim()) {
            return text.trim()
          }
        }

        const clonedElement = element.cloneNode(true)
        const excludeElements = clonedElement.querySelectorAll('[data-testid="message_action_bar"], [data-testid="suggest_message_list"]')
        excludeElements.forEach(el => el.remove())
        return clonedElement.textContent || clonedElement.innerText || ''
      }

      const textElement = this.findWithFallback(
        this.config.selectors.text,
        this.domain === 'claude.ai' ? CLAUDE_FALLBACK_SELECTORS.text : CHATGPT_FALLBACK_SELECTORS.text,
        (selector) => element.querySelector(selector)
      )

      if (textElement) {
        const text = textElement.textContent || ''
        if (text) {
          return text
        }
      }

      return element.textContent || ''
    } catch (e) {
      console.error('[ChatTOC] Error extracting text:', e)
      return ''
    }
  }

  getMessageId(element, index) {
    const dataId = element.getAttribute('data-message-id') ||
      element.getAttribute('data-id') ||
      element.id

    if (dataId) {
      return `${this.domain}-${dataId}`
    }

    return `${this.domain}-idx-${index}`
  }

  parseMessages(settings) {
    const messages = []
    const elements = this.getMessageElements()

    for (let i = 0; i < elements.length; i++) {
      const element = elements[i]
      const isUser = this.isUserMessage(element)

      const text = this.extractText(element)
      const cleanedText = cleanText(text)

      if (cleanedText.length < settings.rules.minLength) {
        continue
      }

      const preview = truncateText(cleanedText, settings.rules.maxLength || PREVIEW_MAX_LENGTH)
      const role = isUser ? 'user' : 'assistant'
      const messageId = this.getMessageId(element, i)

      messages.push({
        id: messageId,
        index: messages.length + 1,
        role,
        text: cleanedText,
        preview,
        element,
        bookmarked: false,
        searchText: cleanedText.toLowerCase()
      })
    }

    return messages
  }
}

// ==================== 侧边栏 UI ====================
class SidebarUI {
  constructor(options) {
    this.host = options.host
    this.siteName = options.siteName || 'AI'
    this.siteIcon = options.siteIcon || '🤖'
    this.siteIconUrl = ''
    this.onScrollToMessage = options.onScrollToMessage
    this.onToggleBookmark = options.onToggleBookmark
    this.onVisibilityChange = options.onVisibilityChange
    this.onCollapseChange = options.onCollapseChange
    this.onRefresh = options.onRefresh
    this.onExportJson = options.onExportJson
    this.onExportMarkdown = options.onExportMarkdown
    this.onOpenSettings = options.onOpenSettings

    this.position = options.position || 'right'
    this.width = options.width || 380
    this.theme = options.theme || 'auto'
    this.language = options.language || 'auto'
    this.userLabel = options.userLabel || '用户'
    this.assistantLabel = options.assistantLabel || 'Assistant'
    this.labelsCustomized = options.labelsCustomized || false
    this.roleStyleEnabled = options.roleStyleEnabled === true
    this.messages = []
    this.filteredEntries = []
    this.searchQuery = ''
    this.filterRole = null
    this.filterBookmarked = false
    this.currentVisibleMessageId = null
    this.sidebarVisible = true
    this.collapsed = false
    this.status = 'loading'
    this.listScrollIndex = 0
    this.exportMenuOpen = false
    this.resizeState = null
    this.dragState = null
    this.ignoreNextToggleClick = false
    this.collapsedTogglePos = { x: null, y: null }
    this.i18n = I18N[resolveLanguage(this.language)] || I18N.zh
    this.resolveSiteIcon(this.siteIcon)

    this.applyRoleStyle()
    this.init()
  }

  init() {
    this.shadow = this.host.attachShadow({ mode: 'open' })
    this.renderShadow()
  }

  getStyles() {
    return `
        :host {
          position: fixed;
          top: 0;
          right: 0;
          height: 100vh;
          width: var(--sidebar-width, 380px);
          z-index: 2147483647;
          font-family: "IBM Plex Sans", "Segoe UI", sans-serif;
          color: #1f2933;
          pointer-events: auto;
        }

        :host([data-theme="dark"]) .sidebar {
          background: linear-gradient(180deg, rgba(15, 23, 42, 0.96) 0%, rgba(2, 6, 23, 0.96) 100%);
          border-left-color: rgba(148, 163, 184, 0.2);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__header,
        :host([data-theme="dark"]) .sidebar__search,
        :host([data-theme="dark"]) .sidebar__footer {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.2);
        }

        :host([data-theme="dark"]) .sidebar__icon-btn {
          background: rgba(148, 163, 184, 0.18);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__search-field {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.3);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__search-field input {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__filter-btn {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.3);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__filter-btn.is-active {
          background: #e2e8f0;
          color: #0f172a;
          border-color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__message {
          background: rgba(15, 23, 42, 0.92);
          border-color: rgba(148, 163, 184, 0.16);
        }

        :host([data-theme="dark"]) .sidebar__message:hover {
          background: rgba(30, 41, 59, 0.9);
        }

        :host([data-theme="dark"]) .sidebar__message-header,
        :host([data-theme="dark"]) .sidebar__message-preview,
        :host([data-theme="dark"]) .sidebar__count {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__message-index {
          color: #f8fafc;
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message {
          --message-tint: rgba(148, 163, 184, 0.08);
          border-left: 3px solid transparent;
          background: linear-gradient(90deg, var(--message-tint), rgba(15, 23, 42, 0.92));
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-user {
          --message-tint: rgba(59, 130, 246, 0.18);
          border-left-color: rgba(96, 165, 250, 0.45);
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-assistant {
          --message-tint: rgba(16, 185, 129, 0.18);
          border-left-color: rgba(52, 211, 153, 0.45);
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-user .sidebar__message-role {
          color: #93c5fd;
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-assistant .sidebar__message-role {
          color: #6ee7b7;
        }

        :host([data-theme="dark"]) .sidebar__export-menu {
          background: #0f172a;
          border-color: rgba(148, 163, 184, 0.3);
        }

        :host([data-theme="dark"]) .sidebar__export-item {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__status {
          background: rgba(15, 23, 42, 0.95);
          border-color: rgba(148, 163, 184, 0.25);
          color: #e2e8f0;
        }

        *,
        *::before,
        *::after {
          box-sizing: border-box;
        }

        :host([data-state="hidden"]) {
          width: 0;
        }

        :host([data-state="collapsed"]) {
          width: 44px;
        }

        .sidebar {
          height: 100%;
          width: 100%;
          display: flex;
          flex-direction: column;
          background: linear-gradient(180deg, rgba(250, 250, 249, 0.98) 0%, rgba(244, 245, 247, 0.98) 100%);
          border-left: 1px solid rgba(15, 23, 42, 0.12);
          box-shadow: -12px 0 24px rgba(15, 23, 42, 0.12);
          overflow: hidden;
        }

        .sidebar__header {
          position: relative;
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 14px 16px 10px;
          background: rgba(255, 255, 255, 0.7);
          border-bottom: 1px solid rgba(15, 23, 42, 0.08);
        }

        .sidebar__site {
          display: flex;
          align-items: center;
          gap: 8px;
          font-weight: 600;
          font-size: 14px;
          letter-spacing: 0.2px;
          color: #0f172a;
        }

        .sidebar__site-icon {
          width: 18px;
          height: 18px;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          font-size: 16px;
          line-height: 1;
          border-radius: 4px;
        }

        .sidebar__site-icon.is-image {
          background-size: cover;
          background-position: center;
          background-repeat: no-repeat;
        }

        .sidebar__site button {
          all: unset;
          cursor: pointer;
          display: inline-flex;
          align-items: center;
          gap: 8px;
          padding: 6px 8px;
          border-radius: 8px;
          background: rgba(15, 23, 42, 0.04);
        }

        .sidebar__header-actions {
          display: flex;
          gap: 8px;
        }

        .sidebar__icon-btn {
          width: 30px;
          height: 30px;
          border-radius: 8px;
          border: none;
          background: rgba(15, 23, 42, 0.06);
          color: #0f172a;
          cursor: pointer;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          transition: background 0.15s ease;
        }

        .sidebar__icon-btn:hover {
          background: rgba(15, 23, 42, 0.12);
        }

        .sidebar__icon-btn.is-primary {
          background: rgba(15, 23, 42, 0.14);
        }

        .sidebar__export-menu {
          position: absolute;
          top: 54px;
          right: 16px;
          background: #fff;
          border: 1px solid rgba(15, 23, 42, 0.12);
          border-radius: 10px;
          box-shadow: 0 16px 24px rgba(15, 23, 42, 0.16);
          padding: 6px;
          display: none;
          z-index: 5;
        }

        .sidebar__export-menu.is-open {
          display: grid;
          gap: 4px;
        }

        .sidebar__export-item {
          border: none;
          background: transparent;
          padding: 8px 10px;
          border-radius: 8px;
          cursor: pointer;
          font-size: 12px;
          color: #0f172a;
          text-align: left;
        }

        .sidebar__export-item:hover {
          background: rgba(15, 23, 42, 0.08);
        }

        .sidebar__search {
          padding: 12px 16px;
          border-bottom: 1px solid rgba(15, 23, 42, 0.06);
          background: rgba(250, 250, 249, 0.9);
          display: flex;
          flex-direction: column;
          gap: 10px;
        }

        .sidebar__search-field {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 8px 10px;
          border-radius: 10px;
          border: 1px solid rgba(15, 23, 42, 0.12);
          background: #fff;
        }

        .sidebar__search-field input {
          border: none;
          outline: none;
          width: 100%;
          font-size: 13px;
          background: transparent;
        }

        .sidebar__search-clear {
          border: none;
          background: none;
          font-size: 16px;
          cursor: pointer;
          color: #94a3b8;
        }

        .sidebar__filters {
          display: flex;
          flex-wrap: wrap;
          gap: 6px;
        }

        .sidebar__filter-btn {
          padding: 6px 10px;
          border-radius: 999px;
          border: 1px solid rgba(15, 23, 42, 0.1);
          background: #fff;
          font-size: 12px;
          cursor: pointer;
          color: #0f172a;
          transition: all 0.15s ease;
        }

        .sidebar__filter-btn.is-active {
          background: #0f172a;
          color: #fff;
          border-color: #0f172a;
        }

        .sidebar__list {
          flex: 1;
          position: relative;
          height: calc(100vh - 300px);
        }

        .sidebar__list-viewport {
          height: 100%;
          overflow: auto;
          position: relative;
        }

        .sidebar__list-spacer {
          height: 0;
        }

        .sidebar__list-items {
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
        }

        .sidebar__message {
          height: ${VIRTUAL_ROW_HEIGHT}px;
          padding: 10px 14px;
          display: flex;
          flex-direction: column;
          justify-content: center;
          gap: 6px;
          border-bottom: 1px solid rgba(15, 23, 42, 0.06);
          cursor: pointer;
          background: rgba(255, 255, 255, 0.9);
          transition: background 0.15s ease;
          position: absolute;
          left: 0;
          right: 0;
        }

        .sidebar__message:hover {
          background: rgba(226, 232, 240, 0.8);
        }

        :host([data-role-style="true"]) .sidebar__message {
          --message-tint: rgba(15, 23, 42, 0.04);
          border-left: 3px solid transparent;
          background: linear-gradient(90deg, var(--message-tint), rgba(255, 255, 255, 0.9));
        }

        :host([data-role-style="true"]) .sidebar__message.is-user {
          --message-tint: rgba(59, 130, 246, 0.08);
          border-left-color: rgba(59, 130, 246, 0.35);
        }

        :host([data-role-style="true"]) .sidebar__message.is-assistant {
          --message-tint: rgba(16, 185, 129, 0.08);
          border-left-color: rgba(16, 185, 129, 0.35);
        }

        :host([data-role-style="true"]) .sidebar__message.is-user .sidebar__message-role {
          color: #2563eb;
        }

        :host([data-role-style="true"]) .sidebar__message.is-assistant .sidebar__message-role {
          color: #059669;
        }

        .sidebar__message.is-active {
          background: rgba(59, 130, 246, 0.15);
          border-left: 3px solid #3b82f6;
        }

        .sidebar__message-header {
          display: flex;
          align-items: center;
          justify-content: space-between;
          font-size: 12px;
          color: #475569;
        }

        .sidebar__message-meta {
          display: inline-flex;
          align-items: center;
          gap: 8px;
        }

        .sidebar__message-index {
          font-weight: 600;
          color: #0f172a;
        }

        .sidebar__message-role {
          display: inline-flex;
          align-items: center;
          gap: 6px;
        }

        .sidebar__message-preview {
          font-size: 12px;
          color: #1f2933;
          line-height: 1.4;
          display: -webkit-box;
          -webkit-line-clamp: 2;
          -webkit-box-orient: vertical;
          overflow: hidden;
        }

        .sidebar__bookmark-btn {
          border: none;
          background: none;
          cursor: pointer;
          font-size: 14px;
          color: #cbd5f5;
        }

        .sidebar__bookmark-btn.is-active {
          color: #f59e0b;
        }

        .sidebar__status {
          position: absolute;
          top: 20px;
          left: 12px;
          right: 12px;
          padding: 16px;
          border-radius: 12px;
          background: rgba(255, 255, 255, 0.96);
          border: 1px solid rgba(15, 23, 42, 0.08);
          text-align: center;
          font-size: 13px;
          color: #475569;
          display: none;
        }

        .sidebar__status.is-visible {
          display: block;
        }

        .sidebar__footer {
          padding: 12px 16px;
          border-top: 1px solid rgba(15, 23, 42, 0.08);
          display: flex;
          align-items: center;
          justify-content: space-between;
          background: rgba(255, 255, 255, 0.9);
        }


        .sidebar__count {
          font-size: 12px;
          color: #64748b;
        }

        mark {
          background: rgba(251, 191, 36, 0.4);
          color: inherit;
          padding: 0 2px;
          border-radius: 3px;
        }

        .sidebar__floating-toggle,
        .sidebar__collapsed-toggle {
          position: fixed;
          right: 12px;
          width: 40px;
          height: 40px;
          border-radius: 12px;
          border: 1px solid rgba(15, 23, 42, 0.1);
          background: rgba(255, 255, 255, 0.95);
          box-shadow: 0 8px 24px rgba(15, 23, 42, 0.2);
          cursor: pointer;
          display: none;
          align-items: center;
          justify-content: center;
        }

        .sidebar__floating-toggle {
          top: 80px;
        }

        .sidebar__collapsed-toggle {
          top: 20px;
        }

        :host([data-state="hidden"]) .sidebar {
          display: none;
        }

        :host([data-state="hidden"]) .sidebar__floating-toggle {
          display: inline-flex;
        }

        :host([data-state="collapsed"]) .sidebar {
          display: none;
        }

        :host([data-state="collapsed"]) .sidebar__collapsed-toggle {
          display: inline-flex;
        }

        .sidebar__resize-handle {
          position: absolute;
          left: 0;
          top: 0;
          bottom: 0;
          width: 6px;
          cursor: ew-resize;
          background: transparent;
          z-index: 2;
        }

        .sidebar__resize-handle:hover {
          background: rgba(15, 23, 42, 0.08);
        }

        @media (max-width: 720px) {
          :host {
            width: 100vw;
          }

          :host([data-state="collapsed"]) {
            width: 44px;
          }
        }
    `
  }

  getTemplate() {
    return `
      <div class="sidebar">
        <div class="sidebar__resize-handle" data-action="resize"></div>
        <header class="sidebar__header">
          <div class="sidebar__site">
            <button type="button" class="sidebar__site-button" title="${this.i18n.siteStatus}">
              <span class="sidebar__site-icon">${this.siteIcon}</span>
              <span class="sidebar__site-name">${this.siteName}</span>
            </button>
          </div>
          <div class="sidebar__header-actions">
            <button class="sidebar__icon-btn" data-action="refresh" title="${this.i18n.refresh}" aria-label="${this.i18n.refresh}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M21 12a9 9 0 1 1-2.64-6.36"></path>
                <path d="M21 3v6h-6"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="export" title="${this.i18n.export}" aria-label="${this.i18n.export}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M12 3v12"></path>
                <path d="M7 10l5 5 5-5"></path>
                <path d="M4 21h16"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="settings" title="${this.i18n.settings}" aria-label="${this.i18n.settings}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="3"></circle>
                <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 1 1-2.83 2.83l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 1 1-4 0v-.09a1.65 1.65 0 0 0-1-1.51 1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 1 1-2.83-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 1 1 0-4h.09a1.65 1.65 0 0 0 1.51-1 1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 1 1 2.83-2.83l.06.06a1.65 1.65 0 0 0 1.82.33h.01A1.65 1.65 0 0 0 9 3.09V3a2 2 0 1 1 4 0v.09c0 .68.39 1.3 1 1.51a1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 1 1 2.83 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82v.01c.21.61.83 1 1.51 1H21a2 2 0 1 1 0 4h-.09c-.68 0-1.3.39-1.51 1z"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="collapse" title="${this.i18n.collapse}" aria-label="${this.i18n.collapse}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M4 7h16M4 12h10M4 17h16" />
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="close" title="${this.i18n.close}" aria-label="${this.i18n.close}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M6 6l12 12M6 18L18 6" />
              </svg>
            </button>
          </div>
        </header>

        <section class="sidebar__search">
          <div class="sidebar__search-field">
            <span class="sidebar__search-icon">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="11" cy="11" r="7"></circle>
                <path d="M21 21l-4.3-4.3"></path>
              </svg>
            </span>
            <input type="text" class="sidebar__search-input" placeholder="${this.i18n.searchPlaceholder}" />
            <button type="button" class="sidebar__search-clear" title="${this.i18n.clear}">×</button>
          </div>
          <div class="sidebar__filters">
            <button class="sidebar__filter-btn" data-role="all">${this.i18n.filterAll}</button>
            <button class="sidebar__filter-btn" data-role="user">${this.i18n.filterUser}</button>
            <button class="sidebar__filter-btn" data-role="assistant">${this.i18n.filterAssistant}</button>
            <button class="sidebar__filter-btn" data-bookmark="true">${this.i18n.filterBookmark}</button>
          </div>
        </section>

        <section class="sidebar__list">
          <div class="sidebar__list-viewport">
            <div class="sidebar__list-spacer"></div>
            <div class="sidebar__list-items"></div>
          </div>
          <div class="sidebar__status"></div>
        </section>

        <footer class="sidebar__footer">
          <div class="sidebar__count">0 / 0</div>
        </footer>
      </div>

      <button class="sidebar__floating-toggle" data-action="open" title="${this.i18n.open}">⛶</button>
      <button class="sidebar__collapsed-toggle" data-action="expand" title="${this.i18n.expand}">⛶</button>
      <div class="sidebar__export-menu" aria-label="${this.i18n.exportMenu}">
        <button class="sidebar__export-item" data-action="export-json" type="button">${this.i18n.exportJson}</button>
        <button class="sidebar__export-item" data-action="export-markdown" type="button">${this.i18n.exportMarkdown}</button>
      </div>
    `
  }

  cacheElements() {
    this.container = this.shadow.querySelector('.sidebar')
    this.searchInput = this.shadow.querySelector('.sidebar__search-input')
    this.searchClear = this.shadow.querySelector('.sidebar__search-clear')
    this.filterButtons = Array.from(this.shadow.querySelectorAll('.sidebar__filter-btn'))
    this.listViewport = this.shadow.querySelector('.sidebar__list-viewport')
    this.listSpacer = this.shadow.querySelector('.sidebar__list-spacer')
    this.listItems = this.shadow.querySelector('.sidebar__list-items')
    this.statusEl = this.shadow.querySelector('.sidebar__status')
    this.countEl = this.shadow.querySelector('.sidebar__count')
    this.exportMenu = this.shadow.querySelector('.sidebar__export-menu')
    this.resizeHandle = this.shadow.querySelector('.sidebar__resize-handle')
    this.collapsedToggleBtn = this.shadow.querySelector('.sidebar__collapsed-toggle')
  }

  bindEvents() {
    this.shadow.addEventListener('click', (event) => {
      const target = event.target
      const targetElement = target instanceof Element ? target : null
      const button = targetElement?.closest('button')
        || (event.composedPath?.().find((node) => node instanceof HTMLButtonElement) || null)

      if (button?.dataset?.action === 'refresh') {
        this.onRefresh?.()
        return
      }

      if (button?.dataset?.action === 'export') {
        this.toggleExportMenu()
        return
      }

      if (button?.dataset?.action === 'export-json') {
        this.onExportJson?.()
        this.closeExportMenu()
        return
      }

      if (button?.dataset?.action === 'export-markdown') {
        this.onExportMarkdown?.()
        this.closeExportMenu()
        return
      }

      if (button?.dataset?.action === 'settings') {
        this.onOpenSettings?.()
        return
      }

      if (button?.dataset?.action === 'close') {
        this.setSidebarVisible(false)
        return
      }

      if (button?.dataset?.action === 'collapse') {
        this.setCollapsed(true)
        return
      }

      if (button?.dataset?.action === 'open') {
        this.setSidebarVisible(true)
        return
      }

      if (button?.dataset?.action === 'expand') {
        if (this.ignoreNextToggleClick) {
          this.ignoreNextToggleClick = false
          return
        }
        this.setCollapsed(false)
        return
      }

      if (button?.classList.contains('sidebar__search-clear')) {
        this.updateSearch('')
        this.searchInput.value = ''
        this.searchInput.focus()
        return
      }

      if (button?.classList.contains('sidebar__bookmark-btn')) {
        event.stopPropagation()
        const id = button.dataset.id
        if (id && this.onToggleBookmark) {
          this.onToggleBookmark(id)
        }
        return
      }

      if (button?.classList.contains('sidebar__filter-btn')) {
        const role = button.dataset.role
        const bookmark = button.dataset.bookmark === 'true'
        if (role) {
          this.setRoleFilter(role)
        } else if (bookmark) {
          this.toggleBookmarkFilter()
        }
        return
      }

      const item = targetElement?.closest('.sidebar__message')
      if (item?.dataset?.id) {
        this.onScrollToMessage?.(item.dataset.id)
      }

      const exportMenuTarget = targetElement?.closest('.sidebar__export-menu')
        || button?.closest?.('.sidebar__export-menu')
      if (!exportMenuTarget && button?.dataset?.action !== 'export') {
        this.closeExportMenu()
      }
    })

    this.resizeHandle?.addEventListener('mousedown', (event) => {
      event.preventDefault()
      this.startResize(event.clientX)
    })

    this.collapsedToggleBtn?.addEventListener('mousedown', (event) => {
      event.preventDefault()
      this.startCollapsedToggleDrag(event)
    })

    this.searchInput.addEventListener('input', debounce((event) => {
      this.updateSearch(event.target.value)
    }, 150))

    this.listViewport.addEventListener('scroll', () => {
      this.renderList()
      this.updateListVisibleIndex()
    })

    this.boundKeydown = (event) => this.handleKeydown(event)
    window.addEventListener('keydown', this.boundKeydown, true)

    const resizeObserver = new ResizeObserver(() => this.renderList())
    resizeObserver.observe(this.listViewport)
    this.resizeObserver = resizeObserver

    this.setWidth(this.width)
    this.applyTheme()
    this.applyCollapsedTogglePosition()
  }

  toggleExportMenu() {
    this.exportMenuOpen = !this.exportMenuOpen
    if (this.exportMenu) {
      this.exportMenu.classList.toggle('is-open', this.exportMenuOpen)
    }
  }

  closeExportMenu() {
    this.exportMenuOpen = false
    if (this.exportMenu) {
      this.exportMenu.classList.remove('is-open')
    }
  }

  handleKeydown(event) {
    const isMeta = event.metaKey || event.ctrlKey
    if (isMeta && event.key.toLowerCase() === 'k') {
      event.preventDefault()
      if (!this.sidebarVisible) {
        this.setSidebarVisible(true)
      }
      if (this.collapsed) {
        this.setCollapsed(false)
      }
      this.searchInput.focus()
      this.searchInput.select()
      return
    }

    if (!this.sidebarVisible) {
      return
    }

    if (isMeta && event.key.toLowerCase() === 'b') {
      if (!isEditableElement(event.target) && this.currentVisibleMessageId) {
        event.preventDefault()
        this.onToggleBookmark?.(this.currentVisibleMessageId)
      }
      return
    }

    if (event.altKey && (event.key === 'ArrowUp' || event.key === 'ArrowDown')) {
      if (!isEditableElement(event.target)) {
        event.preventDefault()
        const direction = event.key === 'ArrowUp' ? -1 : 1
        this.navigateMessage(direction)
      }
      return
    }

    if (event.key === 'Escape') {
      if (this.searchInput.value) {
        this.updateSearch('')
        this.searchInput.value = ''
        return
      }
      this.setSidebarVisible(false)
    }
  }

  navigateMessage(direction) {
    const messages = this.getFilteredMessages()
    if (messages.length === 0) return

    const currentIndex = messages.findIndex((msg) => msg.id === this.currentVisibleMessageId)
    const nextIndex = currentIndex === -1
      ? (direction > 0 ? 0 : messages.length - 1)
      : Math.min(messages.length - 1, Math.max(0, currentIndex + direction))

    const target = messages[nextIndex]
    if (target) {
      this.onScrollToMessage?.(target.id)
    }
  }

  setRoleFilter(role) {
    if (role === 'all') {
      this.filterRole = null
    } else {
      this.filterRole = role
    }
    this.renderFilters()
    this.applyFilters()
  }

  toggleBookmarkFilter() {
    this.filterBookmarked = !this.filterBookmarked
    this.renderFilters()
    this.applyFilters()
  }

  updateSearch(value) {
    this.searchQuery = value
    this.applyFilters()
  }

  applyFilters() {
    const query = this.searchQuery.trim().toLowerCase()

    const filtered = this.messages.filter((message) => {
      if (this.filterRole && message.role !== this.filterRole) {
        return false
      }
      if (this.filterBookmarked && !message.bookmarked) {
        return false
      }
      if (query && !message.searchText.includes(query)) {
        return false
      }
      return true
    })

    this.filteredEntries = filtered.map((message) => ({ type: 'message', message }))
    this.renderList()
    this.updateListVisibleIndex()
  }

  renderFilters() {
    this.filterButtons.forEach((button) => {
      if (button.dataset.role === 'all') {
        button.classList.toggle('is-active', this.filterRole === null)
      }
      if (button.dataset.role === 'user') {
        button.classList.toggle('is-active', this.filterRole === 'user')
      }
      if (button.dataset.role === 'assistant') {
        button.classList.toggle('is-active', this.filterRole === 'assistant')
      }
      if (button.dataset.bookmark === 'true') {
        button.classList.toggle('is-active', this.filterBookmarked)
      }
    })
  }

  renderList() {
    const entries = this.filteredEntries
    const totalHeight = entries.length * VIRTUAL_ROW_HEIGHT
    this.listSpacer.style.height = `${totalHeight}px`

    const scrollTop = this.listViewport.scrollTop
    const viewportHeight = this.listViewport.clientHeight
    const startIndex = Math.max(0, Math.floor(scrollTop / VIRTUAL_ROW_HEIGHT) - VIRTUAL_OVERSCAN)
    const visibleCount = Math.ceil(viewportHeight / VIRTUAL_ROW_HEIGHT) + VIRTUAL_OVERSCAN * 2
    const endIndex = Math.min(entries.length, startIndex + visibleCount)

    const items = []
    for (let i = startIndex; i < endIndex; i++) {
      const entry = entries[i]
      if (!entry) continue

      const message = entry.message
      const isActive = message.id === this.currentVisibleMessageId
      const roleClass = message.role === 'user' ? 'is-user' : 'is-assistant'
      const roleLabel = message.role === 'user'
        ? `👤 ${this.resolveRoleLabel('user')}`
        : `🤖 ${this.resolveRoleLabel('assistant')}`
      const highlighted = highlightText(message.preview, this.searchQuery)

      items.push(`
        <div class="sidebar__message ${roleClass} ${isActive ? 'is-active' : ''}" data-id="${message.id}" style="transform: translateY(${i * VIRTUAL_ROW_HEIGHT}px);" title="${escapeHtml(message.text)}">
          <div class="sidebar__message-header">
            <div class="sidebar__message-meta">
              <span class="sidebar__message-index">#${message.index}</span>
              <span class="sidebar__message-role">${roleLabel}</span>
            </div>
            <button class="sidebar__bookmark-btn ${message.bookmarked ? 'is-active' : ''}" data-id="${message.id}" title="${this.i18n.bookmark}" aria-pressed="${message.bookmarked}">★</button>
          </div>
          <div class="sidebar__message-preview">${highlighted}</div>
        </div>
      `)
    }

    this.listItems.innerHTML = items.join('')

    this.listItems.style.transform = 'translateZ(0)'

    this.renderStatus()
  }

  renderStatus() {
    let message = ''

    if (this.status === 'loading') {
      message = this.i18n.loading
    } else if (this.status === 'unsupported') {
      message = this.i18n.unsupported
    } else if (this.status === 'degraded') {
      message = this.i18n.degraded
    } else if (this.filteredEntries.length === 0) {
      message = this.messages.length === 0 ? this.i18n.empty : this.i18n.emptyFiltered
    }

    if (message) {
      this.statusEl.textContent = message
      this.statusEl.classList.add('is-visible')
    } else {
      this.statusEl.classList.remove('is-visible')
    }
  }

  updateListVisibleIndex() {
    const entries = this.filteredEntries
    if (entries.length === 0) {
      this.listScrollIndex = 0
      this.updateFooter()
      return
    }

    const scrollTop = this.listViewport.scrollTop
    const startIndex = Math.floor(scrollTop / VIRTUAL_ROW_HEIGHT)

    let visibleIndex = 0
    for (let i = startIndex; i < entries.length; i++) {
      const entry = entries[i]
      if (entry?.type === 'message') {
        visibleIndex = entry.message.index
        break
      }
    }

    this.listScrollIndex = visibleIndex
    this.updateFooter()
  }

  updateFooter() {
    const total = this.messages.length
    const current = this.listScrollIndex || 0
    this.countEl.textContent = `${current} / ${total}`
  }

  render() {
    this.renderFilters()
    this.applyFilters()
    this.updateFooter()
  }

  setWidth(width) {
    const clamped = Math.max(SIDEBAR_MIN_WIDTH, Math.min(SIDEBAR_MAX_WIDTH, width))
    this.width = clamped
    this.host.style.setProperty('--sidebar-width', `${clamped}px`)
  }

  setTheme(theme) {
    this.theme = theme || 'auto'
    this.applyTheme()
  }

  setRoleStyleEnabled(enabled) {
    this.roleStyleEnabled = Boolean(enabled)
    this.applyRoleStyle()
  }

  applyTheme() {
    const theme = this.theme === 'auto'
      ? (window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light')
      : this.theme
    this.host.setAttribute('data-theme', theme)
  }

  applyRoleStyle() {
    if (this.roleStyleEnabled) {
      this.host.setAttribute('data-role-style', 'true')
    } else {
      this.host.removeAttribute('data-role-style')
    }
  }

  startCollapsedToggleDrag(event) {
    if (!this.collapsedToggleBtn) return

    const rect = this.collapsedToggleBtn.getBoundingClientRect()
    this.dragState = {
      offsetX: event.clientX - rect.left,
      offsetY: event.clientY - rect.top,
      moved: false
    }

    const handleMove = (moveEvent) => {
      if (!this.dragState || !this.collapsedToggleBtn) return

      const buttonWidth = rect.width || 40
      const buttonHeight = rect.height || 40
      const padding = 8
      const maxX = window.innerWidth - buttonWidth - padding
      const maxY = window.innerHeight - buttonHeight - padding

      const nextX = Math.min(maxX, Math.max(padding, moveEvent.clientX - this.dragState.offsetX))
      const nextY = Math.min(maxY, Math.max(padding, moveEvent.clientY - this.dragState.offsetY))

      this.collapsedTogglePos = { x: nextX, y: nextY }
      this.applyCollapsedTogglePosition()

      if (!this.dragState.moved) {
        const movedDistance = Math.abs(moveEvent.clientX - event.clientX) + Math.abs(moveEvent.clientY - event.clientY)
        if (movedDistance > 4) {
          this.dragState.moved = true
        }
      }
    }

    const handleUp = () => {
      if (this.dragState?.moved) {
        this.ignoreNextToggleClick = true
      }
      this.dragState = null
      document.body.style.cursor = ''
      window.removeEventListener('mousemove', handleMove)
      window.removeEventListener('mouseup', handleUp)
    }

    document.body.style.cursor = 'grabbing'
    window.addEventListener('mousemove', handleMove)
    window.addEventListener('mouseup', handleUp)
  }

  applyCollapsedTogglePosition() {
    if (!this.collapsedToggleBtn) return
    if (this.collapsedTogglePos.x === null || this.collapsedTogglePos.y === null) return

    this.collapsedToggleBtn.style.left = `${this.collapsedTogglePos.x}px`
    this.collapsedToggleBtn.style.top = `${this.collapsedTogglePos.y}px`
    this.collapsedToggleBtn.style.right = 'auto'
  }

  setMessages(messages) {
    this.messages = (messages || []).map((message) => ({
      ...message,
      bookmarked: normalizeBookmarked(message.bookmarked)
    }))
    this.applyFilters()
  }

  setLabels(labels) {
    if (labels?.user) {
      this.userLabel = labels.user
    }
    if (labels?.assistant) {
      this.assistantLabel = labels.assistant
    }
    this.renderList()
  }

  setLabelsCustomized(customized) {
    this.labelsCustomized = customized
    this.renderList()
  }

  resolveRoleLabel(role) {
    if (this.labelsCustomized) {
      return role === 'user' ? this.userLabel : this.assistantLabel
    }
    return role === 'user' ? this.i18n.userLabel : this.i18n.assistantLabel
  }

  setLanguage(language) {
    this.language = language || 'auto'
    this.i18n = I18N[resolveLanguage(this.language)] || I18N.zh
    this.renderShadow()
  }

  renderShadow() {
    if (!this.shadow) return
    this.host.textContent = ''
    this.shadow.innerHTML = this.getTemplate()
    const style = document.createElement('style')
    style.textContent = this.getStyles()
    this.shadow.prepend(style)
    this.cacheElements()
    this.applySiteIcon()
    this.bindEvents()
    this.render()
  }

  setStatus(status) {
    this.status = status
    this.renderStatus()
  }

  setSiteInfo({ name, icon }) {
    if (name) {
      this.siteName = name
      const nameEl = this.shadow.querySelector('.sidebar__site-name')
      if (nameEl) nameEl.textContent = name
    }
    this.resolveSiteIcon(icon)
    this.applySiteIcon()
  }

  resolveSiteIcon(icon) {
    const faviconUrl = getFaviconUrl()
    if (faviconUrl) {
      this.siteIconUrl = faviconUrl
      this.siteIcon = ''
      return
    }
    this.siteIconUrl = ''
    this.siteIcon = icon || '🤖'
  }

  applySiteIcon() {
    const iconEl = this.shadow?.querySelector('.sidebar__site-icon')
    if (!iconEl) return

    if (this.siteIconUrl) {
      iconEl.textContent = ''
      iconEl.classList.add('is-image')
      iconEl.style.backgroundImage = `url("${this.siteIconUrl}")`
    } else {
      iconEl.classList.remove('is-image')
      iconEl.style.backgroundImage = ''
      iconEl.textContent = this.siteIcon || '🤖'
    }
  }

  setCurrentVisibleMessageId(id) {
    this.currentVisibleMessageId = id
    this.renderList()
  }

  setSidebarVisible(visible) {
    this.sidebarVisible = visible
    if (visible) {
      this.host.setAttribute('data-state', this.collapsed ? 'collapsed' : 'expanded')
    } else {
      this.host.setAttribute('data-state', 'hidden')
    }
    if (this.onVisibilityChange) {
      this.onVisibilityChange(visible)
    }
  }

  setCollapsed(collapsed) {
    this.collapsed = collapsed
    if (collapsed) {
      this.host.setAttribute('data-state', 'collapsed')
    } else {
      this.host.setAttribute('data-state', 'expanded')
    }
    if (this.onCollapseChange) {
      this.onCollapseChange(collapsed)
    }
  }

  getFilteredMessages() {
    return this.filteredEntries
      .filter(entry => entry.type === 'message')
      .map(entry => entry.message)
  }

  startResize(startX) {
    if (!this.sidebarVisible || this.collapsed) return

    this.resizeState = {
      startX,
      startWidth: this.width
    }

    const handleMove = (event) => {
      if (!this.resizeState) return
      const delta = this.position === 'left'
        ? event.clientX - this.resizeState.startX
        : this.resizeState.startX - event.clientX
      this.setWidth(this.resizeState.startWidth + delta)
    }

    const handleUp = () => {
      this.resizeState = null
      document.body.style.cursor = ''
      window.removeEventListener('mousemove', handleMove)
      window.removeEventListener('mouseup', handleUp)
    }

    document.body.style.cursor = 'ew-resize'
    window.addEventListener('mousemove', handleMove)
    window.addEventListener('mouseup', handleUp)
  }

  destroy() {
    this.resizeObserver?.disconnect()
    if (this.boundKeydown) {
      window.removeEventListener('keydown', this.boundKeydown, true)
    }
  }
}

// ==================== 主类：ChatTOC ====================
class ChatTOC {
  constructor() {
    this.parser = null
    this.observer = null
    this.settings = null
    this.messages = []
    this.sidebar = null
    this.panelContainer = null
    this.settingsUnsubscribe = null
    this.retryTimeoutId = null
    this.scrollSpyTimer = null
    this.boundScrollHandler = null
    this.bookmarks = {}
    this.conversationKey = ''
    this.sidebarVisible = true
    this.sidebarCollapsed = false
    this.currentVisibleMessageId = null
    this.exporting = false
    this.useSidePanel = false
    this.panelStatus = 'loading'
    this.siteInfo = null
    this.panelUpdateTimer = null
    this.panelMessageBound = false
  }

  async init() {
    // Init start.

    const domain = window.location.hostname
    // Current domain captured.

    this.settings = await StorageManager.getSettings()
    // Settings loaded.

    if (!this.settings.general.enabled) {
      // Extension disabled.
      return
    }

    const isPlatformEnabled = this.settings.platforms[domain] ?? false
    const customSite = this.settings.customSites.find(site =>
      site.domain === domain && site.enabled
    )

    // Platform enabled status checked.

    if (!isPlatformEnabled && !customSite) {
      // Domain not enabled.
      return
    }

    this.parser = new ParserAdapter(domain)

    if (!this.parser.isSupported() && customSite) {
      this.parser.setCustomConfig({
        site_name: customSite.site_name,
        domain: customSite.domain,
        selectors: customSite.selectors
      })
    }

    const siteInfo = SITE_LABELS[domain] || { name: this.parser?.config?.site_name || domain, icon: '🤖' }
    this.conversationKey = this.buildConversationKey(domain)
    this.labelsCustomized = this.isLabelsCustomized(this.settings.labels)
    this.siteInfo = siteInfo
    this.useSidePanel = Boolean(chrome?.runtime?.getManifest?.().side_panel)

    await this.loadLocalState()
    this.bindPanelMessaging()
    if (!this.useSidePanel) {
      this.injectPanel(siteInfo)
    } else {
      this.panelStatus = 'loading'
      this.schedulePanelUpdate()
    }

    if (!this.parser.isSupported()) {
      console.warn('[ChatTOC] 当前域名不支持:', domain)
      this.sidebar?.setStatus('unsupported')
      this.panelStatus = 'unsupported'
      this.schedulePanelUpdate()
      return
    }

    // Parser initialized.

    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', () => {
        // DOMContentLoaded fired.
        this.startWithRetry()
      })
    } else {
      // DOM ready.
      this.startWithRetry()
    }

    this.settingsUnsubscribe = StorageManager.onSettingsChange((newSettings) => {
      // Settings updated.
      this.settings = newSettings
      this.update()
    })
  }

  buildConversationKey(domain) {
    const path = window.location.pathname || '/'
    return `${domain}:${path}`
  }

  bindPanelMessaging() {
    if (this.panelMessageBound || !chrome?.runtime?.onMessage) return
    this.panelMessageBound = true

    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (!message?.type) return

      if (message.type === 'CHATToc_GET_STATE') {
        sendResponse({ ok: true, state: this.getPanelState() })
        return true
      }

      if (message.type === 'CHATToc_SCROLL_TO') {
        if (message.messageId) {
          this.scrollToMessageById(message.messageId)
        }
        return
      }

      if (message.type === 'CHATToc_TOGGLE_BOOKMARK') {
        if (message.messageId) {
          this.toggleBookmark(message.messageId).then((bookmarked) => {
            sendResponse({ ok: true, bookmarked })
          })
          return true
        }
        return
      }

      if (message.type === 'CHATToc_REFRESH') {
        this.parseMessages()
        return
      }

      if (message.type === 'CHATToc_EXPORT') {
        this.exportMessages(message.format)
      }
    })
  }

  schedulePanelUpdate() {
    if (!this.useSidePanel) return
    if (this.panelUpdateTimer) {
      clearTimeout(this.panelUpdateTimer)
    }
    this.panelUpdateTimer = window.setTimeout(() => {
      this.panelUpdateTimer = null
      this.notifyPanelState()
    }, 120)
  }

  notifyPanelState() {
    if (!this.useSidePanel || !chrome?.runtime?.sendMessage) return
    chrome.runtime.sendMessage({ type: 'CHATToc_STATE', state: this.getPanelState() })
  }

  getPanelState() {
    const faviconUrl = getFaviconUrl()
    const baseSiteInfo = this.siteInfo || { name: window.location.hostname, icon: '🤖' }
    return {
      status: this.panelStatus || 'loading',
      siteInfo: {
        ...baseSiteInfo,
        iconUrl: faviconUrl || ''
      },
      messages: this.serializeMessages(this.messages),
      currentVisibleMessageId: this.currentVisibleMessageId || null,
      settings: this.settings
        ? {
          appearance: this.settings.appearance,
          general: this.settings.general,
          labels: this.settings.labels
        }
        : null,
      labelsCustomized: this.labelsCustomized || false
    }
  }

  serializeMessages(messages) {
    return (messages || []).map((message) => ({
      id: message.id,
      index: message.index,
      role: message.role,
      text: message.text,
      preview: message.preview,
      bookmarked: message.bookmarked,
      searchText: message.searchText
    }))
  }

  async loadLocalState() {
    const [bookmarks, sidebarState] = await Promise.all([
      LocalStateManager.get(KEY_STORAGE_BOOKMARKS),
      LocalStateManager.get(KEY_STORAGE_SIDEBAR_STATE)
    ])

    this.bookmarks = bookmarks[this.conversationKey] || {}
    const storedSidebar = sidebarState[this.conversationKey] || {}
    this.sidebarVisible = storedSidebar.visible !== false
    this.sidebarCollapsed = storedSidebar.collapsed === true
  }

  startWithRetry(maxRetries = 5, delay = 1000) {
    let retryCount = 0

    const tryStart = () => {
      if (!this.parser) {
        return
      }

      // Attempt start.

      try {
        const container = this.parser.getContainer()
        if (!container) {
          retryCount++
          if (retryCount < maxRetries) {
            // Container not found, retrying.
            this.retryTimeoutId = window.setTimeout(tryStart, delay)
            return
          }
          console.error('[ChatTOC] 达到最大重试次数，容器仍未找到')
          this.sidebar?.setStatus('degraded')
          this.panelStatus = 'degraded'
          this.schedulePanelUpdate()
          return
        }

        // Container found, parsing messages.
        this.parseMessages()
        this.setupObserver()
        this.setupScrollSpy()
      } catch (error) {
        console.error('[ChatTOC] 启动过程中出错:', error)
        this.sidebar?.setStatus('degraded')
        this.panelStatus = 'degraded'
        this.schedulePanelUpdate()
      }
    }

    this.retryTimeoutId = window.setTimeout(tryStart, 500)
  }

  parseMessages() {
    if (!this.parser || !this.settings) {
      console.warn('[ChatTOC] 解析器或设置未初始化')
      return
    }

    try {
      const container = this.parser.getContainer()
      if (!container) {
        console.warn('[ChatTOC] 容器未找到，无法解析消息')
        return
      }

      const elements = this.parser.getMessageElements()
      this.messages = this.parser.parseMessages(this.settings)

      this.messages.forEach((message) => {
        message.bookmarked = Boolean(this.bookmarks[message.id])
      })

      // Parse completed.

      if (this.messages.length === 0 && elements.length > 0) {
        console.warn('[ChatTOC] 未找到任何消息，可能的原因:')
        console.warn('  - 选择器不匹配当前页面结构')
        console.warn('  - 消息长度小于最小阈值:', this.settings.rules.minLength)
        this.sidebar?.setStatus('degraded')
        this.panelStatus = 'degraded'
      } else {
        this.sidebar?.setStatus('ready')
        this.panelStatus = 'ready'
      }

      this.updateSidebar()
      this.schedulePanelUpdate()
    } catch (error) {
      console.error('[ChatTOC] 解析消息时出错:', error)
      this.messages = []
      this.sidebar?.setStatus('degraded')
      this.panelStatus = 'degraded'
      this.updateSidebar()
      this.schedulePanelUpdate()
    }
  }

  injectPanel(siteInfo) {
    if (this.useSidePanel) {
      return
    }
    if (this.panelContainer && document.body.contains(this.panelContainer)) {
      // Panel exists, update list.
      this.updateSidebar()
      return
    }

    if (this.panelContainer) {
      // Panel container exists but not in DOM.
      this.panelContainer = null
    }

    if (!document.body) {
      console.error('[ChatTOC] document.body 不存在，等待 DOM 加载')
      window.setTimeout(() => this.injectPanel(siteInfo), 100)
      return
    }

    try {
      // Injecting sidebar.

      this.cleanupInjectedTextNodes()

      this.panelContainer = document.createElement('div')
      this.panelContainer.id = 'chattoc-root'
      document.body.appendChild(this.panelContainer)

      this.sidebar = new SidebarUI({
        host: this.panelContainer,
        siteName: siteInfo.name,
        siteIcon: siteInfo.icon,
        position: this.settings.appearance.position,
        width: this.settings.appearance.width,
        theme: this.settings.appearance.theme,
        language: this.settings.general.language,
        userLabel: this.settings.labels?.user,
        assistantLabel: this.settings.labels?.assistant,
        labelsCustomized: this.labelsCustomized,
        roleStyleEnabled: this.settings.appearance.roleStyle,
        onScrollToMessage: (messageId) => this.scrollToMessageById(messageId),
        onToggleBookmark: (messageId) => this.toggleBookmark(messageId),
        onVisibilityChange: (visible) => this.saveSidebarState({ visible }),
        onCollapseChange: (collapsed) => this.saveSidebarState({ collapsed }),
        onRefresh: () => this.parseMessages(),
        onExportJson: () => this.exportMessages('json'),
        onExportMarkdown: () => this.exportMessages('markdown'),
        onOpenSettings: () => this.openSettings()
      })

      this.sidebar.setSidebarVisible(this.sidebarVisible)
      if (this.sidebarVisible) {
        this.sidebar.setCollapsed(this.sidebarCollapsed)
      }
      this.sidebar.setStatus('loading')

      // Sidebar injected.
    } catch (error) {
      console.error('[ChatTOC] 注入面板时出错:', error)
      if (this.panelContainer && this.panelContainer.parentNode) {
        this.panelContainer.remove()
      }
      this.panelContainer = null
      this.sidebar = null
    }
  }

  cleanupInjectedTextNodes() {
    if (!document.body) return
    const suspectRegex = /flex:\\s*1;|overflow:\\s*auto;|position:\\s*relative;|background:\\s*(rgba|transparent)/i
    const nodes = Array.from(document.body.childNodes)
    nodes.forEach((node) => {
      if (node.nodeType !== Node.TEXT_NODE) return
      const text = node.textContent || ''
      if (text.length < 30) return
      if (suspectRegex.test(text)) {
        node.remove()
      }
    })
  }

  openSettings() {
    if (chrome?.runtime?.openOptionsPage) {
      chrome.runtime.openOptionsPage()
      return
    }

    const optionsUrl = chrome?.runtime?.getURL
      ? chrome.runtime.getURL('options/index.html')
      : null

    if (optionsUrl) {
      window.open(optionsUrl, '_blank')
    }
  }

  exportMessages(format) {
    if (this.exporting) {
      return
    }

    if (!this.messages.length) {
      alert('未找到任何消息内容')
      return
    }

    this.exporting = true
    window.setTimeout(() => {
      this.exporting = false
    }, 500)

    const domain = window.location.hostname.replace(/\./g, '_')
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19)
    const baseName = `chat_${domain}_${timestamp}`

    const payload = {
      site: this.parser?.config?.site_name || window.location.hostname,
      url: window.location.href,
      exportedAt: new Date().toISOString(),
      messages: this.messages.map((message) => ({
        index: message.index,
        role: message.role,
        content: message.text
      }))
    }

    if (format === 'markdown') {
      const content = this.buildMarkdownFromPayload(payload)
      this.downloadFile(`${baseName}.md`, content, 'text/markdown')
      return
    }

    this.downloadFile(`${baseName}.json`, JSON.stringify(payload, null, 2), 'application/json')
  }

  buildMarkdownFromPayload(payload) {
    const lines = [`# ChatTOC Export

## Metadata

- Site: ${payload.site}
- URL: ${payload.url}
- Exported: ${payload.exportedAt}

## Messages
`]

    payload.messages.forEach((message) => {
      const roleLabel = message.role === 'user' ? 'User' : 'Assistant'
      lines.push(`### ${message.index}. ${roleLabel}

\`\`\`
${message.content}
\`\`\`
`)
    })

    return lines.join('\n')
  }

  downloadFile(filename, content, type) {
    const blob = new Blob([content], { type })
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = filename
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  updateSidebar() {
    if (this.sidebar) {
      this.sidebar.setMessages(this.messages)
      if (this.currentVisibleMessageId) {
        this.sidebar.setCurrentVisibleMessageId(this.currentVisibleMessageId)
      }
    }
    if (this.useSidePanel) {
      this.schedulePanelUpdate()
    }
  }

  setupObserver() {
    if (!this.parser) return

    const container = this.parser.getContainer()
    if (!container) return

    this.observer = new ObserverManager(() => {
      this.parseMessages()
    }, 300)

    this.observer.observe(container, {
      childList: true,
      subtree: true
    })
  }

  setupScrollSpy() {
    this.boundScrollHandler = () => {
      if (this.scrollSpyTimer) {
        clearTimeout(this.scrollSpyTimer)
      }
      this.scrollSpyTimer = setTimeout(() => this.updateActiveFromViewport(), 100)
    }

    window.addEventListener('scroll', this.boundScrollHandler, { passive: true })
    this.updateActiveFromViewport()
  }

  updateActiveFromViewport() {
    if (!this.messages.length) return

    const viewportCenter = window.scrollY + window.innerHeight / 2
    let closestItem = null
    let closestDistance = Infinity

    for (const item of this.messages) {
      if (!item.element) continue
      const rect = item.element.getBoundingClientRect()
      const elementCenter = rect.top + window.scrollY + rect.height / 2

      if (rect.top < window.innerHeight && rect.bottom > 0) {
        const distance = Math.abs(elementCenter - viewportCenter)
        if (distance < closestDistance) {
          closestDistance = distance
          closestItem = item
        }
      }
    }

    if (closestItem && this.currentVisibleMessageId !== closestItem.id) {
      this.currentVisibleMessageId = closestItem.id
      this.sidebar?.setCurrentVisibleMessageId(closestItem.id)
      this.schedulePanelUpdate()
    }
  }

  async saveSidebarState(partial) {
    await LocalStateManager.update(KEY_STORAGE_SIDEBAR_STATE, (state) => {
      const current = state[this.conversationKey] || {}
      state[this.conversationKey] = { ...current, ...partial }
      return state
    })
  }

  async toggleBookmark(messageId) {
    const message = this.messages.find((item) => item.id === messageId)
    if (!message) return

    message.bookmarked = !message.bookmarked
    if (message.bookmarked) {
      this.bookmarks[messageId] = true
    } else {
      delete this.bookmarks[messageId]
    }

    await LocalStateManager.update(KEY_STORAGE_BOOKMARKS, (state) => {
      state[this.conversationKey] = { ...this.bookmarks }
      return state
    })

    this.sidebar?.setMessages(this.messages)
    this.schedulePanelUpdate()
    return message.bookmarked
  }

  scrollToMessageById(messageId) {
    const item = this.messages.find((message) => message.id === messageId)
    if (!item) {
      console.warn('[ChatTOC] 未找到消息项，ID:', messageId)
      return
    }

    this.currentVisibleMessageId = item.id
    this.sidebar?.setCurrentVisibleMessageId(item.id)
    this.schedulePanelUpdate()
    this.scrollToMessage(item)
  }

  scrollToMessage(item) {
    if (!item?.element) {
      console.warn('[ChatTOC] scrollToMessage: item.element 不存在', item)
      return
    }

    let targetElement = item.element
    const elementInDOM = document.body.contains(targetElement)

    try {
      let finalTargetElement = targetElement
      const textSelectors = [
        '[data-message-author-role="user"]',
        '.markdown',
        '[class*="markdown"]',
        '[class*="text"]',
        '[class*="content"]'
      ]

      for (const selector of textSelectors) {
        try {
          const textEl = targetElement.querySelector(selector)
          if (textEl) {
            if (elementInDOM ? document.body.contains(textEl) : true) {
              finalTargetElement = textEl
              break
            }
          }
        } catch (e) {
          continue
        }
      }

      finalTargetElement.scrollIntoView({
        behavior: this.settings?.appearance?.smoothScroll !== false ? 'smooth' : 'auto',
        block: 'start',
        inline: 'nearest'
      })

      if (elementInDOM && targetElement && document.body.contains(targetElement)) {
        const originalBg = targetElement.style.backgroundColor || ''
        const originalTransition = targetElement.style.transition || ''

        targetElement.style.transition = 'background-color 0.3s'
        targetElement.style.backgroundColor = 'rgba(59, 130, 246, 0.3)'

        setTimeout(() => {
          if (targetElement && document.body.contains(targetElement)) {
            targetElement.style.backgroundColor = originalBg
            setTimeout(() => {
              if (targetElement && document.body.contains(targetElement)) {
                targetElement.style.transition = originalTransition
              }
            }, 300)
          }
        }, 1000)
      }
    } catch (error) {
      console.error('[ChatTOC] 滚动到消息时出错:', error)
      try {
        if (targetElement && document.body.contains(targetElement)) {
          targetElement.scrollIntoView({
            behavior: this.settings?.appearance?.smoothScroll !== false ? 'smooth' : 'auto',
            block: 'start'
          })
        }
      } catch (e) {
        console.error('[ChatTOC] 回退滚动也失败:', e)
      }
    }
  }

  update() {
    if (!this.settings?.general.enabled) {
      this.destroy()
      return
    }

    this.labelsCustomized = this.isLabelsCustomized(this.settings.labels)
    this.parseMessages()
    if (this.sidebar && this.settings) {
      this.sidebar.setWidth(this.settings.appearance.width)
      this.sidebar.setLabels(this.settings.labels)
      this.sidebar.setTheme(this.settings.appearance.theme)
      this.sidebar.setLanguage(this.settings.general.language)
      this.sidebar.setLabelsCustomized(this.labelsCustomized)
      this.sidebar.setRoleStyleEnabled(this.settings.appearance.roleStyle)
    }
    this.schedulePanelUpdate()
  }

  isLabelsCustomized(labels) {
    if (!labels) return false
    return labels.user !== defaultSettings.labels.user ||
      labels.assistant !== defaultSettings.labels.assistant
  }

  destroy() {
    if (this.retryTimeoutId !== null) {
      clearTimeout(this.retryTimeoutId)
      this.retryTimeoutId = null
    }

    if (this.scrollSpyTimer) {
      clearTimeout(this.scrollSpyTimer)
      this.scrollSpyTimer = null
    }

    if (this.panelUpdateTimer) {
      clearTimeout(this.panelUpdateTimer)
      this.panelUpdateTimer = null
    }

    if (this.observer) {
      this.observer.disconnect()
      this.observer = null
    }

    if (this.settingsUnsubscribe) {
      this.settingsUnsubscribe()
      this.settingsUnsubscribe = null
    }

    if (this.boundScrollHandler) {
      window.removeEventListener('scroll', this.boundScrollHandler)
      this.boundScrollHandler = null
    }

    if (this.sidebar) {
      this.sidebar.destroy()
      this.sidebar = null
    }

    if (this.panelContainer && this.panelContainer.parentNode) {
      try {
        this.panelContainer.remove()
      } catch (error) {
        console.error('[ChatTOC] 移除 DOM 元素时出错:', error)
      }
      this.panelContainer = null
    }

    this.messages = []
    this.parser = null
    this.settings = null
  }
}

// ==================== 启动 ====================
try {
  // Boot.
  const chatTOC = new ChatTOC()
  chatTOC.init().catch(error => {
    console.error('[ChatTOC] 初始化失败:', error)
    console.error('[ChatTOC] 错误堆栈:', error.stack)
  })
} catch (error) {
  console.error('[ChatTOC] 创建实例失败:', error)
  console.error('[ChatTOC] 错误堆栈:', error.stack)
}
