const STATUS_UNSUPPORTED = 'unsupported'
const STATUS_LOADING = 'loading'

const $app = $('#app')
let sidebar = null
let currentTabId = null
let lastState = null
let isApplyingState = false
const extensionIconUrl = chrome.runtime.getURL('icons/icon32.png')
const pendingBookmarkOverrides = new Map()

function normalizeBookmarked(value) {
  if (value === true || value === false) return value
  if (value === 'true' || value === 1 || value === '1') return true
  return false
}

function buildExportPayload(state, tab) {
  const url = tab?.url || ''
  const hostname = url ? new URL(url).hostname : ''
  return {
    site: state?.siteInfo?.name || hostname || tab?.title || 'Chat',
    url,
    exportedAt: new Date().toISOString(),
    messages: (state?.messages || []).map((message) => ({
      index: message.index,
      role: message.role,
      content: message.text
    }))
  }
}

function buildMarkdownFromPayload(payload) {
  const lines = [`# ChatTOC Export

## Metadata

- Site: ${payload.site}
- URL: ${payload.url}
- Exported: ${payload.exportedAt}

## Messages
`]

  payload.messages.forEach((message) => {
    const roleLabel = message.role === 'user' ? 'User' : 'Assistant'
    lines.push(`### ${message.index}. ${roleLabel}

\`\`\`
${message.content}
\`\`\`
`)
  })

  return lines.join('\n')
}

function downloadFile(filename, content, type) {
  const blob = new Blob([content], { type })
  const url = URL.createObjectURL(blob)
  const $link = $('<a>')
  $link.attr('href', url)
  $link.attr('download', filename)
  $('body').append($link)
  $link[0].click()
  $link.remove()
  URL.revokeObjectURL(url)
}

async function exportFromState(format) {
  if (!lastState?.messages?.length) {
    window.alert('未找到任何消息内容')
    return
  }

  if (!currentTabId) {
    window.alert('未找到当前标签页')
    return
  }

  const tab = await chrome.tabs.get(currentTabId)
  const payload = buildExportPayload(lastState, tab)
  const domain = (tab?.url ? new URL(tab.url).hostname : 'chat').replace(/\./g, '_')
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19)
  const baseName = `chat_${domain}_${timestamp}`

  if (format === 'markdown') {
    const content = buildMarkdownFromPayload(payload)
    downloadFile(`${baseName}.md`, content, 'text/markdown')
    return
  }

  downloadFile(`${baseName}.json`, JSON.stringify(payload, null, 2), 'application/json')
}

function getActiveTabId() {
  return new Promise((resolve) => {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      const tab = tabs && tabs[0]
      resolve(tab?.id || null)
    })
  })
}

function sendToActiveTab(message, callback) {
  if (!currentTabId) {
    callback?.({ ok: false, error: 'no_active_tab' })
    return
  }
  chrome.tabs.sendMessage(currentTabId, message, (response) => {
    if (chrome.runtime.lastError) {
      callback?.({ ok: false, error: chrome.runtime.lastError.message })
      return
    }
    callback?.(response)
  })
}

function ensureSidebar(state) {
  if (sidebar) return
  const SidebarUI = window.ChatTOCSidebarUI
  sidebar = new SidebarUI({
    host: $app[0],
    embedded: true,
    siteName: state?.siteInfo?.name || 'AI',
    siteIcon: state?.siteInfo?.icon || '🤖',
    siteIconUrl: state?.siteInfo?.iconUrl || extensionIconUrl,
    theme: state?.settings?.appearance?.theme || 'auto',
    language: state?.settings?.general?.language || 'auto',
    userLabel: state?.settings?.labels?.user,
    assistantLabel: state?.settings?.labels?.assistant,
    labelsCustomized: Boolean(state?.labelsCustomized),
    roleStyleEnabled: Boolean(state?.settings?.appearance?.roleStyle),
    onScrollToMessage: (messageId) => {
      sendToActiveTab({ type: 'CHATToc_SCROLL_TO', messageId })
    },
    onToggleBookmark: (messageId) => {
      if (lastState?.messages?.length) {
        const updatedMessages = lastState.messages.map((message) => {
          if (message.id !== messageId) {
            return { ...message, bookmarked: normalizeBookmarked(message.bookmarked) }
          }
          const nextValue = !normalizeBookmarked(message.bookmarked)
          pendingBookmarkOverrides.set(messageId, nextValue)
          return { ...message, bookmarked: nextValue }
        })
        lastState = { ...lastState, messages: updatedMessages }
        sidebar.setMessages(updatedMessages)
      }
      sendToActiveTab({ type: 'CHATToc_TOGGLE_BOOKMARK', messageId }, (response) => {
        if (!response?.ok || typeof response.bookmarked !== 'boolean') return
        pendingBookmarkOverrides.delete(messageId)
        if (lastState?.messages?.length) {
          const updatedMessages = lastState.messages.map((message) => (
            message.id === messageId
              ? { ...message, bookmarked: response.bookmarked }
              : message
          ))
          lastState = { ...lastState, messages: updatedMessages }
          sidebar.setMessages(updatedMessages)
        }
      })
    },
    onRefresh: () => {
      sendToActiveTab({ type: 'CHATToc_REFRESH' }, () => {
        requestState()
      })
    },
    onExportJson: () => {
      exportFromState('json')
    },
    onExportMarkdown: () => {
      exportFromState('markdown')
    },
    onOpenSettings: () => {
      if (chrome.runtime.openOptionsPage) {
        chrome.runtime.openOptionsPage()
      }
    }
  })
}

function applyState(state) {
  if (!state) {
    if (!sidebar) {
      ensureSidebar()
    }
    sidebar?.setStatus(STATUS_UNSUPPORTED)
    return
  }

  ensureSidebar(state)
  const mergedMessages = (state.messages || []).map((message) => {
    if (!pendingBookmarkOverrides.has(message.id)) {
      return { ...message, bookmarked: normalizeBookmarked(message.bookmarked) }
    }
    const pendingValue = pendingBookmarkOverrides.get(message.id)
    const incomingValue = normalizeBookmarked(message.bookmarked)
    if (incomingValue === pendingValue) {
      pendingBookmarkOverrides.delete(message.id)
    }
    return { ...message, bookmarked: pendingValue }
  })
  lastState = { ...state, messages: mergedMessages }

  isApplyingState = true
  sidebar.setSiteInfo({
    name: state.siteInfo?.name || 'AI',
    icon: state.siteInfo?.icon || '🤖',
    iconUrl: state.siteInfo?.iconUrl || extensionIconUrl
  })
  sidebar.setMessages(mergedMessages)
  sidebar.setStatus(state.status || STATUS_LOADING)
  sidebar.setCurrentVisibleMessageId(state.currentVisibleMessageId || null)
  if (state.settings?.labels) {
    sidebar.setLabels(state.settings.labels)
  }
  if (typeof state.labelsCustomized === 'boolean') {
    sidebar.setLabelsCustomized(state.labelsCustomized)
  }
  if (state.settings?.appearance?.theme) {
    sidebar.setTheme(state.settings.appearance.theme)
  }
  if (typeof state.settings?.appearance?.roleStyle === 'boolean') {
    sidebar.setRoleStyleEnabled(state.settings.appearance.roleStyle)
  }
  if (state.settings?.general?.language) {
    sidebar.setLanguage(state.settings.general.language)
  }
  isApplyingState = false
}

function requestState() {
  sendToActiveTab({ type: 'CHATToc_GET_STATE' }, (response) => {
    if (!response?.ok) {
      applyState({ status: STATUS_UNSUPPORTED, messages: [] })
      return
    }
    applyState(response.state)
  })
}

function bindIncomingUpdates() {
  chrome.runtime.onMessage.addListener((message, sender) => {
    if (message?.type !== 'CHATToc_STATE') return
    const senderTabId = sender?.tab?.id
    if (!currentTabId || senderTabId !== currentTabId) return
    applyState(message.state)
  })
}

function handleActiveTabChange() {
  getActiveTabId().then((tabId) => {
    if (!tabId || tabId === currentTabId) return
    currentTabId = tabId
    requestState()
  })
}

function bindTabListeners() {
  if (chrome.tabs?.onActivated) {
    chrome.tabs.onActivated.addListener(() => {
      handleActiveTabChange()
    })
  }

  if (chrome.tabs?.onUpdated) {
    chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
      if (tabId !== currentTabId) return
      if (changeInfo.status === 'complete' || changeInfo.url) {
        requestState()
      }
    })
  }

  if (chrome.windows?.onFocusChanged) {
    chrome.windows.onFocusChanged.addListener(() => {
      handleActiveTabChange()
    })
  }
}

async function boot() {
  currentTabId = await getActiveTabId()
  bindIncomingUpdates()
  bindTabListeners()
  requestState()
}

boot()
