/* Sidebar UI for Side Panel */
const VIRTUAL_ROW_HEIGHT = 82
const VIRTUAL_OVERSCAN = 6
const SIDEBAR_MIN_WIDTH = 240
const SIDEBAR_MAX_WIDTH = 520

const I18N = {
  zh: {
    siteStatus: '站点状态',
    collapse: '收起侧边栏',
    close: '关闭侧边栏',
    open: '打开侧边栏',
    expand: '展开侧边栏',
    refresh: '刷新消息',
    export: '导出',
    exportJson: '导出 JSON',
    exportMarkdown: '导出 Markdown',
    exportMenu: '导出菜单',
    settings: '设置',
    clear: '清除',
    bookmark: '收藏',
    searchPlaceholder: '搜索消息内容 / 关键词',
    filterAll: '全部',
    filterUser: '用户',
    filterAssistant: 'AI',
    filterBookmark: '★ 收藏',
    userLabel: '用户',
    assistantLabel: 'Assistant',
    loading: '⏳ 正在解析对话…',
    unsupported: '⚠ 当前页面暂不支持',
    degraded: '⚠ 检测到站点更新，部分功能可能受限',
    empty: '暂无消息',
    emptyFiltered: '没有匹配的消息'
  },
  en: {
    siteStatus: 'Site status',
    collapse: 'Collapse sidebar',
    close: 'Close sidebar',
    open: 'Open sidebar',
    expand: 'Expand sidebar',
    refresh: 'Refresh messages',
    export: 'Export',
    exportJson: 'Export JSON',
    exportMarkdown: 'Export Markdown',
    exportMenu: 'Export menu',
    settings: 'Settings',
    clear: 'Clear',
    bookmark: 'Bookmark',
    searchPlaceholder: 'Search messages / keywords',
    filterAll: 'All',
    filterUser: 'User',
    filterAssistant: 'AI',
    filterBookmark: '★ Favorites',
    userLabel: 'User',
    assistantLabel: 'Assistant',
    loading: '⏳ Parsing conversation…',
    unsupported: '⚠ Page not supported',
    degraded: '⚠ Site update detected, some features limited',
    empty: 'No messages yet',
    emptyFiltered: 'No matching messages'
  }
}

function resolveLanguage(lang) {
  if (lang === 'auto') {
    const locale = navigator.language || 'zh'
    return locale.toLowerCase().startsWith('zh') ? 'zh' : 'en'
  }
  return lang || 'zh'
}

function escapeHtml(text) {
  return $('<div>').text(text).html()
}

function highlightText(text, query) {
  if (!query) return escapeHtml(text)

  const lowerText = text.toLowerCase()
  const lowerQuery = query.toLowerCase()
  const queryLength = query.length
  let result = ''
  let startIndex = 0
  let matchIndex = lowerText.indexOf(lowerQuery)

  while (matchIndex !== -1) {
    result += escapeHtml(text.slice(startIndex, matchIndex))
    result += `<mark>${escapeHtml(text.slice(matchIndex, matchIndex + queryLength))}</mark>`
    startIndex = matchIndex + queryLength
    matchIndex = lowerText.indexOf(lowerQuery, startIndex)
  }

  result += escapeHtml(text.slice(startIndex))
  return result
}

function debounce(fn, delay) {
  let timer = null
  return (...args) => {
    if (timer) {
      clearTimeout(timer)
    }
    timer = window.setTimeout(() => {
      fn(...args)
      timer = null
    }, delay)
  }
}

function isEditableElement(target) {
  if (!target) return false
  const tagName = target.tagName?.toLowerCase()
  return tagName === 'input' || tagName === 'textarea' || target.isContentEditable
}

function getFaviconUrl() {
  const selectors = [
    'link[rel="icon"]',
    'link[rel="shortcut icon"]',
    'link[rel="apple-touch-icon"]',
    'link[rel="apple-touch-icon-precomposed"]'
  ]

  for (const selector of selectors) {
    const link = $(selector).get(0)
    const href = link ? $(link).attr('href') : null
    if (!href) continue
    try {
      return new URL(href, window.location.origin).href
    } catch (error) {
      return href
    }
  }

  return ''
}

function normalizeBookmarked(value) {
  if (value === true || value === false) return value
  if (value === 'true' || value === 1 || value === '1') return true
  return false
}

class SidebarUI {
  constructor(options) {
    this.host = options.host
    this.siteName = options.siteName || 'AI'
    this.siteIcon = options.siteIcon || '🤖'
    this.siteIconUrl = ''
    this.siteIconUrlOverride = options.siteIconUrl || ''
    this.onScrollToMessage = options.onScrollToMessage
    this.onToggleBookmark = options.onToggleBookmark
    this.onVisibilityChange = options.onVisibilityChange
    this.onCollapseChange = options.onCollapseChange
    this.onRefresh = options.onRefresh
    this.onExportJson = options.onExportJson
    this.onExportMarkdown = options.onExportMarkdown
    this.onOpenSettings = options.onOpenSettings

    this.position = options.position || 'right'
    this.width = options.width || 380
    this.theme = options.theme || 'auto'
    this.language = options.language || 'auto'
    this.userLabel = options.userLabel || '用户'
    this.assistantLabel = options.assistantLabel || 'Assistant'
    this.labelsCustomized = options.labelsCustomized || false
    this.roleStyleEnabled = options.roleStyleEnabled === true
    this.embedded = options.embedded === true
    this.messages = []
    this.filteredEntries = []
    this.searchQuery = ''
    this.filterRole = null
    this.filterBookmarked = false
    this.currentVisibleMessageId = null
    this.sidebarVisible = true
    this.collapsed = false
    this.status = 'loading'
    this.listScrollIndex = 0
    this.exportMenuOpen = false
    this.resizeState = null
    this.dragState = null
    this.ignoreNextToggleClick = false
    this.collapsedTogglePos = { x: null, y: null }
    this.i18n = I18N[resolveLanguage(this.language)] || I18N.zh
    this.resolveSiteIcon(this.siteIcon)

    if (this.embedded) {
      $(this.host).attr('data-embedded', 'true')
    }

    this.applyRoleStyle()
    this.init()
  }

  init() {
    this.shadow = this.host.attachShadow({ mode: 'open' })
    this.renderShadow()
  }

  getStyles() {
    return `
        :host {
          position: fixed;
          top: 0;
          right: 0;
          height: 100vh;
          width: var(--sidebar-width, 380px);
          z-index: 2147483647;
          font-family: "IBM Plex Sans", "Segoe UI", sans-serif;
          color: #1f2933;
          pointer-events: auto;
        }

        :host([data-embedded="true"]) {
          position: relative;
          width: 100%;
          height: 100%;
          z-index: auto;
        }

        :host([data-embedded="true"]) .sidebar {
          border-left: none;
          box-shadow: none;
        }

        :host([data-embedded="true"]) .sidebar__resize-handle,
        :host([data-embedded="true"]) .sidebar__floating-toggle,
        :host([data-embedded="true"]) .sidebar__collapsed-toggle,
        :host([data-embedded="true"]) .sidebar__icon-btn[data-action="close"],
        :host([data-embedded="true"]) .sidebar__icon-btn[data-action="collapse"] {
          display: none;
        }

        :host([data-theme="dark"]) .sidebar {
          background: linear-gradient(180deg, rgba(15, 23, 42, 0.96) 0%, rgba(2, 6, 23, 0.96) 100%);
          border-left-color: rgba(148, 163, 184, 0.2);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__header,
        :host([data-theme="dark"]) .sidebar__search,
        :host([data-theme="dark"]) .sidebar__footer {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.2);
        }

        :host([data-theme="dark"]) .sidebar__icon-btn {
          background: rgba(148, 163, 184, 0.18);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__search-field {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.3);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__search-field input {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__filter-btn {
          background: rgba(15, 23, 42, 0.9);
          border-color: rgba(148, 163, 184, 0.3);
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__filter-btn.is-active {
          background: #e2e8f0;
          color: #0f172a;
          border-color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__message {
          background: rgba(15, 23, 42, 0.92);
          border-color: rgba(148, 163, 184, 0.16);
        }

        :host([data-theme="dark"]) .sidebar__message:hover {
          background: rgba(30, 41, 59, 0.9);
        }

        :host([data-theme="dark"]) .sidebar__message-header,
        :host([data-theme="dark"]) .sidebar__message-preview,
        :host([data-theme="dark"]) .sidebar__count {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__message-index {
          color: #f8fafc;
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message {
          --message-tint: rgba(148, 163, 184, 0.08);
          border-left: 3px solid transparent;
          background: linear-gradient(90deg, var(--message-tint), rgba(15, 23, 42, 0.92));
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-user {
          --message-tint: rgba(59, 130, 246, 0.18);
          border-left-color: rgba(96, 165, 250, 0.45);
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-assistant {
          --message-tint: rgba(16, 185, 129, 0.18);
          border-left-color: rgba(52, 211, 153, 0.45);
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-user .sidebar__message-role {
          color: #93c5fd;
        }

        :host([data-theme="dark"][data-role-style="true"]) .sidebar__message.is-assistant .sidebar__message-role {
          color: #6ee7b7;
        }


        :host([data-theme="dark"]) .sidebar__export-menu {
          background: #0f172a;
          border-color: rgba(148, 163, 184, 0.3);
        }

        :host([data-theme="dark"]) .sidebar__export-item {
          color: #e2e8f0;
        }

        :host([data-theme="dark"]) .sidebar__status {
          background: rgba(15, 23, 42, 0.95);
          border-color: rgba(148, 163, 184, 0.25);
          color: #e2e8f0;
        }

        *,
        *::before,
        *::after {
          box-sizing: border-box;
        }

        :host([data-state="hidden"]) {
          width: 0;
        }

        :host([data-state="collapsed"]) {
          width: 44px;
        }

        .sidebar {
          height: 100%;
          width: 100%;
          display: flex;
          flex-direction: column;
          background: linear-gradient(180deg, rgba(250, 250, 249, 0.98) 0%, rgba(244, 245, 247, 0.98) 100%);
          border-left: 1px solid rgba(15, 23, 42, 0.12);
          box-shadow: -12px 0 24px rgba(15, 23, 42, 0.12);
          overflow: hidden;
        }

        .sidebar__header {
          position: relative;
          display: flex;
          align-items: center;
          justify-content: space-between;
          padding: 14px 16px 10px;
          background: rgba(255, 255, 255, 0.7);
          border-bottom: 1px solid rgba(15, 23, 42, 0.08);
        }

        .sidebar__site {
          display: flex;
          align-items: center;
          gap: 8px;
          font-weight: 600;
          font-size: 14px;
          letter-spacing: 0.2px;
          color: #0f172a;
        }

        .sidebar__site-icon {
          width: 18px;
          height: 18px;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          font-size: 16px;
          line-height: 1;
          border-radius: 4px;
        }

        .sidebar__site-icon.is-image {
          background-size: cover;
          background-position: center;
          background-repeat: no-repeat;
        }

        .sidebar__site button {
          all: unset;
          cursor: pointer;
          display: inline-flex;
          align-items: center;
          gap: 8px;
          padding: 6px 8px;
          border-radius: 8px;
          background: rgba(15, 23, 42, 0.04);
        }

        .sidebar__header-actions {
          display: flex;
          gap: 8px;
        }

        .sidebar__icon-btn {
          width: 30px;
          height: 30px;
          border-radius: 8px;
          border: none;
          background: rgba(15, 23, 42, 0.06);
          color: #0f172a;
          cursor: pointer;
          display: inline-flex;
          align-items: center;
          justify-content: center;
          transition: background 0.15s ease;
        }

        .sidebar__icon-btn:hover {
          background: rgba(15, 23, 42, 0.12);
        }

        .sidebar__icon-btn.is-primary {
          background: rgba(15, 23, 42, 0.14);
        }

        .sidebar__export-menu {
          position: absolute;
          top: 54px;
          right: 16px;
          background: #fff;
          border: 1px solid rgba(15, 23, 42, 0.12);
          border-radius: 10px;
          box-shadow: 0 16px 24px rgba(15, 23, 42, 0.16);
          padding: 6px;
          display: none;
          z-index: 5;
        }

        .sidebar__export-menu.is-open {
          display: grid;
          gap: 4px;
        }

        .sidebar__export-item {
          border: none;
          background: transparent;
          padding: 8px 10px;
          border-radius: 8px;
          cursor: pointer;
          font-size: 12px;
          color: #0f172a;
          text-align: left;
        }

        .sidebar__export-item:hover {
          background: rgba(15, 23, 42, 0.08);
        }

        .sidebar__search {
          padding: 12px 16px;
          border-bottom: 1px solid rgba(15, 23, 42, 0.06);
          background: rgba(250, 250, 249, 0.9);
          display: flex;
          flex-direction: column;
          gap: 10px;
        }

        .sidebar__search-field {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 8px 10px;
          border-radius: 10px;
          border: 1px solid rgba(15, 23, 42, 0.12);
          background: #fff;
        }

        .sidebar__search-field input {
          border: none;
          outline: none;
          width: 100%;
          font-size: 13px;
          background: transparent;
        }

        .sidebar__search-clear {
          border: none;
          background: none;
          font-size: 16px;
          cursor: pointer;
          color: #94a3b8;
        }

        .sidebar__filters {
          display: flex;
          flex-wrap: wrap;
          gap: 6px;
        }

        .sidebar__filter-btn {
          padding: 6px 10px;
          border-radius: 999px;
          border: 1px solid rgba(15, 23, 42, 0.1);
          background: #fff;
          font-size: 12px;
          cursor: pointer;
          color: #0f172a;
          transition: all 0.15s ease;
        }

        .sidebar__filter-btn.is-active {
          background: #0f172a;
          color: #fff;
          border-color: #0f172a;
        }

        .sidebar__list {
          flex: 1;
          position: relative;
          height: calc(100vh - 300px);
        }

        .sidebar__list-viewport {
          height: 100%;
          overflow: auto;
          position: relative;
        }

        .sidebar__list-spacer {
          height: 0;
        }

        .sidebar__list-items {
          position: absolute;
          top: 0;
          left: 0;
          right: 0;
        }

        .sidebar__message {
          height: ${VIRTUAL_ROW_HEIGHT}px;
          padding: 10px 14px;
          display: flex;
          flex-direction: column;
          justify-content: center;
          gap: 6px;
          border-bottom: 1px solid rgba(15, 23, 42, 0.06);
          cursor: pointer;
          background: rgba(255, 255, 255, 0.9);
          transition: background 0.15s ease;
          position: absolute;
          left: 0;
          right: 0;
        }

        .sidebar__message:hover {
          background: rgba(226, 232, 240, 0.8);
        }

        :host([data-role-style="true"]) .sidebar__message {
          --message-tint: rgba(15, 23, 42, 0.04);
          border-left: 3px solid transparent;
          background: linear-gradient(90deg, var(--message-tint), rgba(255, 255, 255, 0.9));
        }

        :host([data-role-style="true"]) .sidebar__message.is-user {
          --message-tint: rgba(59, 130, 246, 0.08);
          border-left-color: rgba(59, 130, 246, 0.35);
        }

        :host([data-role-style="true"]) .sidebar__message.is-assistant {
          --message-tint: rgba(16, 185, 129, 0.08);
          border-left-color: rgba(16, 185, 129, 0.35);
        }

        :host([data-role-style="true"]) .sidebar__message.is-user .sidebar__message-role {
          color: #2563eb;
        }

        :host([data-role-style="true"]) .sidebar__message.is-assistant .sidebar__message-role {
          color: #059669;
        }

        .sidebar__message.is-active {
          background: rgba(59, 130, 246, 0.15);
          border-left: 3px solid #3b82f6;
        }

        .sidebar__message-header {
          display: flex;
          align-items: center;
          justify-content: space-between;
          font-size: 12px;
          color: #475569;
        }

        .sidebar__message-meta {
          display: inline-flex;
          align-items: center;
          gap: 8px;
        }

        .sidebar__message-index {
          font-weight: 600;
          color: #0f172a;
        }

        .sidebar__message-role {
          display: inline-flex;
          align-items: center;
          gap: 6px;
        }

        .sidebar__message-preview {
          font-size: 12px;
          color: #1f2933;
          line-height: 1.4;
          display: -webkit-box;
          -webkit-line-clamp: 2;
          -webkit-box-orient: vertical;
          overflow: hidden;
        }

        .sidebar__bookmark-btn {
          border: none;
          background: none;
          cursor: pointer;
          font-size: 14px;
          color: #cbd5f5;
        }

        .sidebar__bookmark-btn.is-active {
          color: #f59e0b;
        }

        .sidebar__status {
          position: absolute;
          top: 20px;
          left: 12px;
          right: 12px;
          padding: 16px;
          border-radius: 12px;
          background: rgba(255, 255, 255, 0.96);
          border: 1px solid rgba(15, 23, 42, 0.08);
          text-align: center;
          font-size: 13px;
          color: #475569;
          display: none;
        }

        .sidebar__status.is-visible {
          display: block;
        }

        .sidebar__footer {
          padding: 12px 16px;
          border-top: 1px solid rgba(15, 23, 42, 0.08);
          display: flex;
          align-items: center;
          justify-content: space-between;
          background: rgba(255, 255, 255, 0.9);
        }


        .sidebar__count {
          font-size: 12px;
          color: #64748b;
        }

        mark {
          background: rgba(251, 191, 36, 0.4);
          color: inherit;
          padding: 0 2px;
          border-radius: 3px;
        }

        .sidebar__floating-toggle,
        .sidebar__collapsed-toggle {
          position: fixed;
          right: 12px;
          width: 40px;
          height: 40px;
          border-radius: 12px;
          border: 1px solid rgba(15, 23, 42, 0.1);
          background: rgba(255, 255, 255, 0.95);
          box-shadow: 0 8px 24px rgba(15, 23, 42, 0.2);
          cursor: pointer;
          display: none;
          align-items: center;
          justify-content: center;
        }

        .sidebar__floating-toggle {
          top: 80px;
        }

        .sidebar__collapsed-toggle {
          top: 20px;
        }

        :host([data-state="hidden"]) .sidebar {
          display: none;
        }

        :host([data-state="hidden"]) .sidebar__floating-toggle {
          display: inline-flex;
        }

        :host([data-state="collapsed"]) .sidebar {
          display: none;
        }

        :host([data-state="collapsed"]) .sidebar__collapsed-toggle {
          display: inline-flex;
        }

        .sidebar__resize-handle {
          position: absolute;
          left: 0;
          top: 0;
          bottom: 0;
          width: 6px;
          cursor: ew-resize;
          background: transparent;
          z-index: 2;
        }

        .sidebar__resize-handle:hover {
          background: rgba(15, 23, 42, 0.08);
        }

        @media (max-width: 720px) {
          :host {
            width: 100vw;
          }

          :host([data-state="collapsed"]) {
            width: 44px;
          }
        }
    `
  }

  getTemplate() {
    return `
      <div class="sidebar">
        <div class="sidebar__resize-handle" data-action="resize"></div>
        <header class="sidebar__header">
          <div class="sidebar__site">
            <button type="button" class="sidebar__site-button" title="${this.i18n.siteStatus}">
              <span class="sidebar__site-icon">${this.siteIcon}</span>
              <span class="sidebar__site-name">${this.siteName}</span>
            </button>
          </div>
          <div class="sidebar__header-actions">
            <button class="sidebar__icon-btn" data-action="refresh" title="${this.i18n.refresh}" aria-label="${this.i18n.refresh}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M21 12a9 9 0 1 1-2.64-6.36"></path>
                <path d="M21 3v6h-6"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="export" title="${this.i18n.export}" aria-label="${this.i18n.export}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M12 3v12"></path>
                <path d="M7 10l5 5 5-5"></path>
                <path d="M4 21h16"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="settings" title="${this.i18n.settings}" aria-label="${this.i18n.settings}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="3"></circle>
                <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 1 1-2.83 2.83l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 1 1-4 0v-.09a1.65 1.65 0 0 0-1-1.51 1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 1 1-2.83-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 1 1 0-4h.09a1.65 1.65 0 0 0 1.51-1 1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 1 1 2.83-2.83l.06.06a1.65 1.65 0 0 0 1.82.33h.01A1.65 1.65 0 0 0 9 3.09V3a2 2 0 1 1 4 0v.09c0 .68.39 1.3 1 1.51a1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 1 1 2.83 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82v.01c.21.61.83 1 1.51 1H21a2 2 0 1 1 0 4h-.09c-.68 0-1.3.39-1.51 1z"></path>
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="collapse" title="${this.i18n.collapse}" aria-label="${this.i18n.collapse}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M4 7h16M4 12h10M4 17h16" />
              </svg>
            </button>
            <button class="sidebar__icon-btn" data-action="close" title="${this.i18n.close}" aria-label="${this.i18n.close}">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M6 6l12 12M6 18L18 6" />
              </svg>
            </button>
          </div>
        </header>

        <section class="sidebar__search">
          <div class="sidebar__search-field">
            <span class="sidebar__search-icon">
              <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="11" cy="11" r="7"></circle>
                <path d="M21 21l-4.3-4.3"></path>
              </svg>
            </span>
            <input type="text" class="sidebar__search-input" placeholder="${this.i18n.searchPlaceholder}" />
            <button type="button" class="sidebar__search-clear" title="${this.i18n.clear}">×</button>
          </div>
          <div class="sidebar__filters">
            <button class="sidebar__filter-btn" data-role="all">${this.i18n.filterAll}</button>
            <button class="sidebar__filter-btn" data-role="user">${this.i18n.filterUser}</button>
            <button class="sidebar__filter-btn" data-role="assistant">${this.i18n.filterAssistant}</button>
            <button class="sidebar__filter-btn" data-bookmark="true">${this.i18n.filterBookmark}</button>
          </div>
        </section>

        <section class="sidebar__list">
          <div class="sidebar__list-viewport">
            <div class="sidebar__list-spacer"></div>
            <div class="sidebar__list-items"></div>
          </div>
          <div class="sidebar__status"></div>
        </section>

        <footer class="sidebar__footer">
          <div class="sidebar__count">0 / 0</div>
        </footer>
      </div>

      <button class="sidebar__floating-toggle" data-action="open" title="${this.i18n.open}">⛶</button>
      <button class="sidebar__collapsed-toggle" data-action="expand" title="${this.i18n.expand}">⛶</button>
      <div class="sidebar__export-menu" aria-label="${this.i18n.exportMenu}">
        <button class="sidebar__export-item" data-action="export-json" type="button">${this.i18n.exportJson}</button>
        <button class="sidebar__export-item" data-action="export-markdown" type="button">${this.i18n.exportMarkdown}</button>
      </div>
    `
  }

  cacheElements() {
    this.$shadow = $(this.shadow)
    this.$container = this.$shadow.find('.sidebar')
    this.$searchInput = this.$shadow.find('.sidebar__search-input')
    this.$searchClear = this.$shadow.find('.sidebar__search-clear')
    this.$filterButtons = this.$shadow.find('.sidebar__filter-btn')
    this.$listViewport = this.$shadow.find('.sidebar__list-viewport')
    this.$listSpacer = this.$shadow.find('.sidebar__list-spacer')
    this.$listItems = this.$shadow.find('.sidebar__list-items')
    this.$statusEl = this.$shadow.find('.sidebar__status')
    this.$countEl = this.$shadow.find('.sidebar__count')
    this.$exportMenu = this.$shadow.find('.sidebar__export-menu')
    this.$resizeHandle = this.$shadow.find('.sidebar__resize-handle')
    this.$collapsedToggleBtn = this.$shadow.find('.sidebar__collapsed-toggle')
  }

  bindEvents() {
    $(this.host).off('click.chatToc')
    this.$resizeHandle.off('mousedown.chatToc')
    this.$collapsedToggleBtn.off('mousedown.chatToc')
    this.$searchInput.off('input.chatToc')
    this.$listViewport.off('scroll.chatToc')
    $(window).off('keydown.chatToc')

    $(this.host).on('click.chatToc', (event) => {
      const originalEvent = event.originalEvent || event
      const path = originalEvent.composedPath?.() || []
      const targetElement = path.find((node) => node instanceof Element) || event.target
      const composedButton = path.find((node) => node instanceof HTMLButtonElement) || null
      const closestButton = targetElement instanceof Element ? $(targetElement).closest('button').get(0) : null
      const button = closestButton || composedButton

      if (button && $(button).data('action') === 'refresh') {
        this.onRefresh?.()
        return
      }

      if (button && $(button).data('action') === 'export') {
        this.toggleExportMenu()
        return
      }

      if (button && $(button).data('action') === 'export-json') {
        this.onExportJson?.()
        this.closeExportMenu()
        return
      }

      if (button && $(button).data('action') === 'export-markdown') {
        this.onExportMarkdown?.()
        this.closeExportMenu()
        return
      }

      if (button && $(button).data('action') === 'settings') {
        this.onOpenSettings?.()
        return
      }

      if (button && $(button).data('action') === 'close') {
        this.setSidebarVisible(false)
        return
      }

      if (button && $(button).data('action') === 'collapse') {
        this.setCollapsed(true)
        return
      }

      if (button && $(button).data('action') === 'open') {
        this.setSidebarVisible(true)
        return
      }

      if (button && $(button).data('action') === 'expand') {
        if (this.ignoreNextToggleClick) {
          this.ignoreNextToggleClick = false
          return
        }
        this.setCollapsed(false)
        return
      }

      if (button && $(button).hasClass('sidebar__search-clear')) {
        this.updateSearch('')
        this.$searchInput.val('')
        this.$searchInput.focus()
        return
      }

      if (button && $(button).hasClass('sidebar__bookmark-btn')) {
        event.stopPropagation()
        const id = $(button).data('id')
        if (id && this.onToggleBookmark) {
          this.onToggleBookmark(id)
        }
        return
      }

      if (button && $(button).hasClass('sidebar__filter-btn')) {
        const role = $(button).data('role')
        const bookmark = $(button).data('bookmark') === true
        if (role) {
          this.setRoleFilter(role)
        } else if (bookmark) {
          this.toggleBookmarkFilter()
        }
        return
      }

      const item = path.find((node) => node instanceof Element && $(node).hasClass('sidebar__message'))
        || (targetElement instanceof Element ? $(targetElement).closest('.sidebar__message').get(0) : null)
      if (item && $(item).data('id')) {
        this.onScrollToMessage?.($(item).data('id'))
      }

      const exportMenuTarget = path.find((node) => node instanceof Element && $(node).hasClass('sidebar__export-menu'))
        || (targetElement instanceof Element ? $(targetElement).closest('.sidebar__export-menu').get(0) : null)
      const buttonMenuTarget = button ? $(button).closest('.sidebar__export-menu').get(0) : null
      if (!exportMenuTarget && !buttonMenuTarget && $(button).data('action') !== 'export') {
        this.closeExportMenu()
      }
    })

    if (this.$resizeHandle.length) {
      this.$resizeHandle.on('mousedown.chatToc', (event) => {
        event.preventDefault()
        this.startResize(event.clientX)
      })
    }

    if (this.$collapsedToggleBtn.length) {
      this.$collapsedToggleBtn.on('mousedown.chatToc', (event) => {
        event.preventDefault()
        this.startCollapsedToggleDrag(event)
      })
    }

    this.$searchInput.on('input.chatToc', debounce((event) => {
      this.updateSearch($(event.target).val())
    }, 150))

    this.$listViewport.on('scroll.chatToc', () => {
      this.renderList()
      this.updateListVisibleIndex()
    })

    this.boundKeydown = (event) => this.handleKeydown(event)
    $(window).on('keydown.chatToc', this.boundKeydown)

    const listViewportEl = this.$listViewport.get(0)
    if (listViewportEl) {
      const resizeObserver = new ResizeObserver(() => this.renderList())
      resizeObserver.observe(listViewportEl)
      this.resizeObserver = resizeObserver
    }

    this.setWidth(this.width)
    this.applyTheme()
    this.applyCollapsedTogglePosition()
  }

  toggleExportMenu() {
    this.exportMenuOpen = !this.exportMenuOpen
    if (this.$exportMenu.length) {
      this.$exportMenu.toggleClass('is-open', this.exportMenuOpen)
    }
  }

  closeExportMenu() {
    this.exportMenuOpen = false
    if (this.$exportMenu.length) {
      this.$exportMenu.removeClass('is-open')
    }
  }

  handleKeydown(event) {
    const isMeta = event.metaKey || event.ctrlKey
    if (isMeta && event.key.toLowerCase() === 'k') {
      event.preventDefault()
      if (!this.sidebarVisible) {
        this.setSidebarVisible(true)
      }
      if (this.collapsed) {
        this.setCollapsed(false)
      }
      this.$searchInput.focus()
      this.$searchInput.get(0)?.select()
      return
    }

    if (!this.sidebarVisible) {
      return
    }

    if (isMeta && event.key.toLowerCase() === 'b') {
      if (!isEditableElement(event.target) && this.currentVisibleMessageId) {
        event.preventDefault()
        this.onToggleBookmark?.(this.currentVisibleMessageId)
      }
      return
    }

    if (event.altKey && (event.key === 'ArrowUp' || event.key === 'ArrowDown')) {
      if (!isEditableElement(event.target)) {
        event.preventDefault()
        const direction = event.key === 'ArrowUp' ? -1 : 1
        this.navigateMessage(direction)
      }
      return
    }

    if (event.key === 'Escape') {
      if (this.$searchInput.val()) {
        this.updateSearch('')
        this.$searchInput.val('')
        return
      }
      this.setSidebarVisible(false)
    }
  }

  navigateMessage(direction) {
    const messages = this.getFilteredMessages()
    if (messages.length === 0) return

    const currentIndex = messages.findIndex((msg) => msg.id === this.currentVisibleMessageId)
    const nextIndex = currentIndex === -1
      ? (direction > 0 ? 0 : messages.length - 1)
      : Math.min(messages.length - 1, Math.max(0, currentIndex + direction))

    const target = messages[nextIndex]
    if (target) {
      this.onScrollToMessage?.(target.id)
    }
  }

  setRoleFilter(role) {
    if (role === 'all') {
      this.filterRole = null
    } else {
      this.filterRole = role
    }
    this.renderFilters()
    this.applyFilters()
  }

  toggleBookmarkFilter() {
    this.filterBookmarked = !this.filterBookmarked
    this.renderFilters()
    this.applyFilters()
  }

  updateSearch(value) {
    this.searchQuery = value
    this.applyFilters()
  }

  applyFilters() {
    const query = this.searchQuery.trim().toLowerCase()

    const filtered = this.messages.filter((message) => {
      if (this.filterRole && message.role !== this.filterRole) {
        return false
      }
      if (this.filterBookmarked && !message.bookmarked) {
        return false
      }
      if (query && !message.searchText.includes(query)) {
        return false
      }
      return true
    })

    this.filteredEntries = filtered.map((message) => ({ type: 'message', message }))
    this.renderList()
    this.updateListVisibleIndex()
  }

  renderFilters() {
    this.$filterButtons.each((_, button) => {
      const $button = $(button)
      if ($button.data('role') === 'all') {
        $button.toggleClass('is-active', this.filterRole === null)
      }
      if ($button.data('role') === 'user') {
        $button.toggleClass('is-active', this.filterRole === 'user')
      }
      if ($button.data('role') === 'assistant') {
        $button.toggleClass('is-active', this.filterRole === 'assistant')
      }
      if ($button.data('bookmark') === true) {
        $button.toggleClass('is-active', this.filterBookmarked)
      }
    })
  }

  renderList() {
    const entries = this.filteredEntries
    const totalHeight = entries.length * VIRTUAL_ROW_HEIGHT
    this.$listSpacer.css('height', `${totalHeight}px`)

    const scrollTop = this.$listViewport.scrollTop()
    const viewportHeight = this.$listViewport.innerHeight()
    const startIndex = Math.max(0, Math.floor(scrollTop / VIRTUAL_ROW_HEIGHT) - VIRTUAL_OVERSCAN)
    const visibleCount = Math.ceil(viewportHeight / VIRTUAL_ROW_HEIGHT) + VIRTUAL_OVERSCAN * 2
    const endIndex = Math.min(entries.length, startIndex + visibleCount)

    const items = []
    for (let i = startIndex; i < endIndex; i++) {
      const entry = entries[i]
      if (!entry) continue

      const message = entry.message
      const isActive = message.id === this.currentVisibleMessageId
      const roleClass = message.role === 'user' ? 'is-user' : 'is-assistant'
      const roleLabel = message.role === 'user'
        ? `👤 ${this.resolveRoleLabel('user')}`
        : `🤖 ${this.resolveRoleLabel('assistant')}`
      const highlighted = highlightText(message.preview, this.searchQuery)

      items.push(`
        <div class="sidebar__message ${roleClass} ${isActive ? 'is-active' : ''}" data-id="${message.id}" style="transform: translateY(${i * VIRTUAL_ROW_HEIGHT}px);" title="${escapeHtml(message.text)}">
          <div class="sidebar__message-header">
            <div class="sidebar__message-meta">
              <span class="sidebar__message-index">#${message.index}</span>
              <span class="sidebar__message-role">${roleLabel}</span>
            </div>
            <button class="sidebar__bookmark-btn ${message.bookmarked ? 'is-active' : ''}" data-id="${message.id}" title="${this.i18n.bookmark}" aria-pressed="${message.bookmarked}">★</button>
          </div>
          <div class="sidebar__message-preview">${highlighted}</div>
        </div>
      `)
    }

    this.$listItems.html(items.join(''))

    this.$listItems.css('transform', 'translateZ(0)')

    this.renderStatus()
  }

  renderStatus() {
    let message = ''

    if (this.status === 'loading') {
      message = this.i18n.loading
    } else if (this.status === 'unsupported') {
      message = this.i18n.unsupported
    } else if (this.status === 'degraded') {
      message = this.i18n.degraded
    } else if (this.filteredEntries.length === 0) {
      message = this.messages.length === 0 ? this.i18n.empty : this.i18n.emptyFiltered
    }

    if (message) {
      this.$statusEl.text(message)
      this.$statusEl.addClass('is-visible')
    } else {
      this.$statusEl.removeClass('is-visible')
    }
  }

  updateListVisibleIndex() {
    const entries = this.filteredEntries
    if (entries.length === 0) {
      this.listScrollIndex = 0
      this.updateFooter()
      return
    }

    const scrollTop = this.$listViewport.scrollTop()
    const startIndex = Math.floor(scrollTop / VIRTUAL_ROW_HEIGHT)

    let visibleIndex = 0
    for (let i = startIndex; i < entries.length; i++) {
      const entry = entries[i]
      if (entry?.type === 'message') {
        visibleIndex = entry.message.index
        break
      }
    }

    this.listScrollIndex = visibleIndex
    this.updateFooter()
  }

  updateFooter() {
    const total = this.messages.length
    const current = this.listScrollIndex || 0
    this.$countEl.text(`${current} / ${total}`)
  }

  render() {
    this.renderFilters()
    this.applyFilters()
    this.updateFooter()
  }

  setWidth(width) {
    if (this.embedded) {
      this.width = width
      $(this.host).css('--sidebar-width', '100%')
      return
    }
    const clamped = Math.max(SIDEBAR_MIN_WIDTH, Math.min(SIDEBAR_MAX_WIDTH, width))
    this.width = clamped
    $(this.host).css('--sidebar-width', `${clamped}px`)
  }

  setTheme(theme) {
    this.theme = theme || 'auto'
    this.applyTheme()
  }

  setRoleStyleEnabled(enabled) {
    this.roleStyleEnabled = Boolean(enabled)
    this.applyRoleStyle()
  }

  applyTheme() {
    const theme = this.theme === 'auto'
      ? (window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light')
      : this.theme
    $(this.host).attr('data-theme', theme)
  }

  applyRoleStyle() {
    if (this.roleStyleEnabled) {
      $(this.host).attr('data-role-style', 'true')
    } else {
      $(this.host).removeAttr('data-role-style')
    }
  }

  startCollapsedToggleDrag(event) {
    if (this.embedded || !this.$collapsedToggleBtn.length) return

    const rect = this.$collapsedToggleBtn.get(0).getBoundingClientRect()
    this.dragState = {
      offsetX: event.clientX - rect.left,
      offsetY: event.clientY - rect.top,
      moved: false
    }

    const handleMove = (moveEvent) => {
      if (!this.dragState || !this.$collapsedToggleBtn.length) return

      const buttonWidth = rect.width || 40
      const buttonHeight = rect.height || 40
      const padding = 8
      const maxX = window.innerWidth - buttonWidth - padding
      const maxY = window.innerHeight - buttonHeight - padding

      const nextX = Math.min(maxX, Math.max(padding, moveEvent.clientX - this.dragState.offsetX))
      const nextY = Math.min(maxY, Math.max(padding, moveEvent.clientY - this.dragState.offsetY))

      this.collapsedTogglePos = { x: nextX, y: nextY }
      this.applyCollapsedTogglePosition()

      if (!this.dragState.moved) {
        const movedDistance = Math.abs(moveEvent.clientX - event.clientX) + Math.abs(moveEvent.clientY - event.clientY)
        if (movedDistance > 4) {
          this.dragState.moved = true
        }
      }
    }

    const handleUp = () => {
      if (this.dragState?.moved) {
        this.ignoreNextToggleClick = true
      }
      this.dragState = null
      $('body').css('cursor', '')
      $(window).off('mousemove', handleMove)
      $(window).off('mouseup', handleUp)
    }

    $('body').css('cursor', 'grabbing')
    $(window).on('mousemove', handleMove)
    $(window).on('mouseup', handleUp)
  }

  applyCollapsedTogglePosition() {
    if (!this.$collapsedToggleBtn.length) return
    if (this.collapsedTogglePos.x === null || this.collapsedTogglePos.y === null) return

    this.$collapsedToggleBtn.css({
      left: `${this.collapsedTogglePos.x}px`,
      top: `${this.collapsedTogglePos.y}px`,
      right: 'auto'
    })
  }

  setMessages(messages) {
    this.messages = (messages || []).map((message) => ({
      ...message,
      bookmarked: normalizeBookmarked(message.bookmarked)
    }))
    this.applyFilters()
  }

  setLabels(labels) {
    if (labels?.user) {
      this.userLabel = labels.user
    }
    if (labels?.assistant) {
      this.assistantLabel = labels.assistant
    }
    this.renderList()
  }

  setLabelsCustomized(customized) {
    this.labelsCustomized = customized
    this.renderList()
  }

  resolveRoleLabel(role) {
    if (this.labelsCustomized) {
      return role === 'user' ? this.userLabel : this.assistantLabel
    }
    return role === 'user' ? this.i18n.userLabel : this.i18n.assistantLabel
  }

  setLanguage(language) {
    const nextLanguage = language || 'auto'
    if (this.language === nextLanguage) {
      return
    }
    this.language = nextLanguage
    this.i18n = I18N[resolveLanguage(this.language)] || I18N.zh
    this.renderShadow()
  }

  renderShadow() {
    if (!this.shadow) return
    $(this.host).empty()
    const $shadowRoot = $(this.shadow)
    $shadowRoot.contents().remove()
    $shadowRoot.append(this.getTemplate())
    const $style = $('<style>').text(this.getStyles())
    $shadowRoot.prepend($style)
    this.cacheElements()
    if (this.searchQuery) {
      this.$searchInput.val(this.searchQuery)
    }
    this.applySiteIcon()
    this.bindEvents()
    this.render()
  }

  setStatus(status) {
    this.status = status
    this.renderStatus()
  }

  setSiteInfo({ name, icon, iconUrl }) {
    if (name) {
      this.siteName = name
      const $nameEl = this.$shadow.find('.sidebar__site-name')
      if ($nameEl.length) $nameEl.text(name)
    }
    if (iconUrl) {
      this.siteIconUrlOverride = iconUrl
    }
    this.resolveSiteIcon(icon)
    this.applySiteIcon()
  }

  resolveSiteIcon(icon) {
    if (this.siteIconUrlOverride) {
      this.siteIconUrl = this.siteIconUrlOverride
      this.siteIcon = ''
      return
    }
    const faviconUrl = getFaviconUrl()
    if (faviconUrl) {
      this.siteIconUrl = faviconUrl
      this.siteIcon = ''
      return
    }
    this.siteIconUrl = ''
    this.siteIcon = icon || '🤖'
  }

  applySiteIcon() {
    const $iconEl = this.$shadow.find('.sidebar__site-icon')
    if (!$iconEl.length) return

    if (this.siteIconUrl) {
      $iconEl.text('')
      $iconEl.addClass('is-image')
      $iconEl.css('background-image', `url("${this.siteIconUrl}")`)
    } else {
      $iconEl.removeClass('is-image')
      $iconEl.css('background-image', '')
      $iconEl.text(this.siteIcon || '🤖')
    }
  }

  setCurrentVisibleMessageId(id) {
    this.currentVisibleMessageId = id
    this.renderList()
  }

  setSidebarVisible(visible) {
    if (this.embedded) {
      this.sidebarVisible = true
      this.collapsed = false
      $(this.host).attr('data-state', 'expanded')
      return
    }
    this.sidebarVisible = visible
    if (visible) {
      $(this.host).attr('data-state', this.collapsed ? 'collapsed' : 'expanded')
    } else {
      $(this.host).attr('data-state', 'hidden')
    }
    if (this.onVisibilityChange) {
      this.onVisibilityChange(visible)
    }
  }

  setCollapsed(collapsed) {
    if (this.embedded) {
      this.collapsed = false
      $(this.host).attr('data-state', 'expanded')
      return
    }
    this.collapsed = collapsed
    if (collapsed) {
      $(this.host).attr('data-state', 'collapsed')
    } else {
      $(this.host).attr('data-state', 'expanded')
    }
    if (this.onCollapseChange) {
      this.onCollapseChange(collapsed)
    }
  }

  getFilteredMessages() {
    return this.filteredEntries
      .filter(entry => entry.type === 'message')
      .map(entry => entry.message)
  }

  startResize(startX) {
    if (this.embedded || !this.sidebarVisible || this.collapsed) return

    this.resizeState = {
      startX,
      startWidth: this.width
    }

    const handleMove = (event) => {
      if (!this.resizeState) return
      const delta = this.position === 'left'
        ? event.clientX - this.resizeState.startX
        : this.resizeState.startX - event.clientX
      this.setWidth(this.resizeState.startWidth + delta)
    }

    const handleUp = () => {
      this.resizeState = null
      $('body').css('cursor', '')
      $(window).off('mousemove', handleMove)
      $(window).off('mouseup', handleUp)
    }

    $('body').css('cursor', 'ew-resize')
    $(window).on('mousemove', handleMove)
    $(window).on('mouseup', handleUp)
  }

  destroy() {
    this.resizeObserver?.disconnect()
    if (this.boundKeydown) {
      $(window).off('keydown', this.boundKeydown)
    }
  }
}

window.ChatTOCSidebarUI = SidebarUI
